/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.auditing;

import com.arsdigita.domain.DataObjectNotFoundException;

import com.arsdigita.kernel.ACSObject;
import com.arsdigita.kernel.User;

import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.metadata.ObjectType;


import java.util.Date;

/**
 * Base class. Provides default functionality for auditing ACSObjects.
 *
 *  <p>
 *    
 *  </p>
 *
 * @author Joseph Bank 
 * @version 1.0
 */

public abstract class AuditedACSObject extends ACSObject implements Audited {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/auditing/AuditedACSObject.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Audit trail.
     */

    private BasicAuditTrail m_audit_trail;

    /**
     * Gets the user who created the object. May be null.
     * @return the user who created the object.
     */

    public User getCreationUser() {
        return m_audit_trail.getCreationUser();
    }

    /**
     * Gets the creation date of the object.
     * @return the creation date.
     */

    public Date getCreationDate() {
        return m_audit_trail.getCreationDate();
    }

    /**
     * Gets the IP address associated with creating an object. May be
     * null.
     * @return the creation IP address.
     */

    public String getCreationIP() {
        return m_audit_trail.getCreationIP();
    }

    /**
     * Gets the user who last modified the object. May be null.
     * @return the last modifying user.
     */

    public User getLastModifiedUser() {
        return m_audit_trail.getLastModifiedUser();
    }

    /**
     * Gets the last modified date.
     * @return the last modified date.
     */

    public Date getLastModifiedDate() {
        return m_audit_trail.getLastModifiedDate();
    }

    /**
     * Gets the last modified IP address. May be null.
     * @return the IP address associated with the last modification.
     */

    public String getLastModifiedIP() {
        return m_audit_trail.getLastModifiedIP();
    }

    /**
     * Initializes with a basic audit trail and an Auditing Observer.
     * This method is called from the DomainObject constructor, so it
     * is invoked whenever a new ACSObject is constructed.
     *
     * @post m_audit_trail != null
     */

    protected void initialize() {
        super.initialize();

        //Get the audit trail for this object

        m_audit_trail = BasicAuditTrail.retrieveForACSObject(this);
        addObserver(new AuditingObserver(m_audit_trail));
    }


    /**
     * Equaivalent to the corresponding ACSObject constructor.
     */

    protected AuditedACSObject(DataObject AuditedACSObjectData) {
        super(AuditedACSObjectData);
    }

    /**
     * Equaivalent to the corresponding ACSObject constructor.
     */

    public AuditedACSObject(String typeName) {
        super(typeName);
    }

    /**
     * Equaivalent to the corresponding ACSObject constructor.
     */
    public AuditedACSObject(ObjectType type) {
        super(type);
    }

    /**
     * Equaivalent to the corresponding ACSObject constructor.
     */

    public AuditedACSObject(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }
}
