/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.messaging;

import java.util.Iterator;
import java.io.File;
import javax.activation.DataHandler;
import com.arsdigita.domain.DataObjectNotFoundException;

/**
 * Test cases for attachments.
 *
 * @version $Id: //core-platform/dev/test/src/com/arsdigita/messaging/AttachmentTest.java#7 $
 */

public class AttachmentTest extends MessageTestCase {

    public static final String versionId = "$Id: //core-platform/dev/test/src/com/arsdigita/messaging/AttachmentTest.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static String subject = "This is the subject";
    private static String body    = "This is the body";
    private static String type    = "text/plain";

    private static String name    = "MyAttachment";
    private static String text    = "An attached message";
    private static String html    = "<p>An attached HTML message</p>";

    final static String TEST_IMAGE =
        "attch.gif";
    final static String TEST_IMAGE_PATH =
        System.getProperty("test.base.dir") + "/com/arsdigita/messaging/" + TEST_IMAGE;
    final static String TEST_IMAGE_TYPE =
        "image/gif";

    final static String TEST_FILE =
        "attch.html";
    final static String TEST_FILE_PATH =
        System.getProperty("test.base.dir") + "/com/arsdigita/messaging/" + TEST_FILE;
    final static String TEST_FILE_TYPE =
        MessageType.TEXT_HTML;

    public AttachmentTest(String name) {
        super(name);
    }

    /**
     * Create a simple message using the various set methods and
     * attach something to it.
     */

    public void testAttachment001() {

        Message msg = new Message(from,subject,body);

        // Attach something simple to the message
        msg.attach(text, name);

        // When we save the message, all attachments will become
        // persistent.
        msg.save();
    }

    /**
     * Try attaching several things and then iterating over them.
     */

    public void testAttachment002() {

        Message msg = new Message(from,subject,body);
        msg.attach(text,name);
        msg.attach(text,name);
        msg.save();

        Iterator parts = msg.getAttachments();

        while (parts.hasNext()) {
            MessagePart part = (MessagePart) parts.next();
            assertEquals(name, part.getName());
        }
    }

    /**
     * Try attaching a part that was created separately.
     */

    public void testAttachment003() {

        MessagePart part = new MessagePart();
        part.setName(name);
        part.setText(text);

        Message msg = new Message(from,subject,body);
        msg.attach(part);
        msg.save();

        assertEquals(1, msg.getAttachmentCount());
    }

    /**
     * Try attaching an HTML message.
     */

    public void testAttachment004() {

        MessagePart part = new MessagePart(name, null);
        part.setContent(html,MessagePart.TEXT_HTML);

        Message msg = new Message(from,subject,body);
        msg.attach(part);
        msg.save();

        assertEquals(1, msg.getAttachmentCount());

    }

    /**
     * Try attaching an image from a file.
     */

    public void testAttachment005() {

        File image = new File(TEST_IMAGE_PATH);

        MessagePart part = new MessagePart();
        part.setContent(image, TEST_IMAGE, "A sample image");

        Message msg = new Message(from,subject,body);
        msg.attach(part);
        msg.save();

        // Verify that we have one attachment
        assertEquals(1, msg.getAttachmentCount());

        // Retrieve it and verify the contents
        part = (MessagePart) msg.getAttachments().next();
        assertEquals(TEST_IMAGE, part.getName());
        assertEquals(TEST_IMAGE_TYPE, part.getContentType());
    }

    /**
     * Try attaching an HTML document from a file.
     */

    public void testAttachment006() {

        File image = new File(TEST_FILE_PATH);

        MessagePart part = new MessagePart();
        part.setContent(image, TEST_FILE, "A sample document");

        Message msg = new Message(from,subject,body);
        msg.attach(part);
        msg.save();

        // Verify that we have one attachment
        assertEquals(1, msg.getAttachmentCount());

        // Retrieve it and verify the contents
        part = (MessagePart) msg.getAttachments().next();
        assertEquals(TEST_FILE, part.getName());
        assertEquals(TEST_FILE_TYPE, part.getContentType());
    }

    /**
     * Main method required to make this test runnable.
     */

    public static void main (String args[]) {
        junit.textui.TestRunner.run(AttachmentTest.class);
    }

}
