/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.ui.admin;

import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SimpleComponent;

import com.arsdigita.domain.DataObjectNotFoundException;

import com.arsdigita.kernel.EmailAddress;
import com.arsdigita.kernel.User;
import com.arsdigita.kernel.UserAuthentication;

import com.arsdigita.xml.Element;

import java.util.Iterator;

/**
 * A Bebop component that holds information on one user. The XML
 * format for this element is:
 *
 * <pre>
 * &lt;userInfo
 *     id
 *     name
 *     email
 *     screenName
 *     URI
 *     memberState&gt;
 *     &lt;additionalEmails&gt;...&lt;/additionalEmails&gt;
 * &lt;/userInfo&gt;
 * </pre>
 *
 * @version $Id: //core-platform/dev/src/com/arsdigita/ui/admin/UserInfo.java#11 $
 */

class UserInfo extends SimpleComponent implements AdminConstants {

    /**
     * The element represent by this component.
     */

    private Element m_elmt;

    private UserBrowsePane m_parent;

    public UserInfo(UserBrowsePane parent) {
        m_parent = parent;
    }

    /**
     * generates userInfo element.
     *
     * @param state The current page state
     * @param parent The parent element in the DOM that we add this element to
     *
     * @author Kevin Scaldeferri 
     */
    public void generateXML(PageState state, Element parent) {

        m_elmt = new Element("admin:userInfo", ADMIN_XML_NS);
        parent.addContent(m_elmt);

        User user = m_parent.getUser(state);

        m_elmt.addAttribute("id", user.getID().toString());
        m_elmt.addAttribute("name", user.getDisplayName());
        m_elmt.addAttribute("screenName", user.getScreenName());
        m_elmt.addAttribute("URI", user.getURI());

        // Add the member state (requires some additional checking)

        addMemberState(user);

        // Build the list of email addresses for this user.  We set a
        // special flag for the primary email address.

        EmailAddress primary = user.getPrimaryEmail();
        addEmail(primary,true);

        Iterator iter = user.getAlternateEmails();
        while (iter.hasNext()) {
            addEmail((EmailAddress) iter.next(), false);
        }

    }

    /**
     * Helper method to generate an email element and add it to the
     * UserInfo element.
     */

    private void addEmail (EmailAddress email,
                           boolean isPrimary)
    {
        Element elmt = new Element("admin:email", ADMIN_XML_NS);
        elmt.addAttribute("address", email.getEmailAddress());

        if (isPrimary) {
            elmt.addAttribute("primary", "t");
        }

        m_elmt.addContent(elmt);
    }

    /**
     * Helper method to add the correct member state to the UserInfo
     * element.  Sets it to one of the following:
     *
     * <ul>
     * <li><b>approved</b> if the user has a valid authentication
     * record</li>
     * </li><b>unauthorized</b> if the user does not have a valid
     * authentication record</li>
     * </ul>
     *
     * <p><b>Note</b>: when user states are implemented this code will
     * change to do a simple lookup of the current user state.
     */

    private void addMemberState (User user) {

        String state = "approved";

        try {
            UserAuthentication.retrieveForUser(user);
        } catch (DataObjectNotFoundException ex) {
            state = "unauthorized";
        }

        m_elmt.addAttribute("memberState", state);
    }

}
