/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search;

import com.arsdigita.util.Assert;
import com.arsdigita.persistence.OID;
import com.arsdigita.kernel.Party;
import java.util.Date;
import java.util.Locale;
import java.net.URL;
import java.math.BigDecimal;

/**
 * The Document interface provides access to the metadata associated 
 * with a single document obtained from the search index. Rather
 * than implementing this interface directly, it is preferrable to
 * subclass BaseDocument since this provides protection against
 * future additions this interface
 *
 * @see com.arsdigita.search.ResultSet
 * @see com.arsdigita.search.BaseDocument
 */
public interface Document {

    /**
     * Gets the unique OID for the domain object
     * referenced by this document
     * @return the unique OID
     */
    OID getOID();

    /**
     * Gets the url for this document
     * @return the document URL
     */
    URL getURL();

    /**
     * Gets the locale to which this object belongs
     *
     * @return the locale of the object
     *
     * @pos $retval != null
     */
    Locale getLocale();

    /**
     * Gets the Title property for the DomainObject
     *
     * @return title of the object
     *
     * @post $retval != null
     */
    String getTitle();

    /**
     * Gets the (optional) summary of the DomainObject
     *
     * @return the object summary, or null
     */
    String getSummary();
    
    /**
     * Gets the (optional) creation date of the DomainObject
     *
     * @return the creation date, or null
     */
    Date getCreationDate();

    /**
     * Gets the (optional) creating party of the DomainObject
     *
     * @return the creation party, or null
     */
    Party getCreationParty();

    /**
     * Gets the (optional) last modification date of the DomainObject
     *
     * @return the modification date, or null
     */
    Date getLastModifiedDate();

    /**
     * Gets the (optional) last modifying party of the DomainObject
     *
     * @return the modification party, or null
     */
    Party getLastModifiedParty();
    
    /**
     * Gets the document score. The range of values returned
     * by this method are dependant on the indexer backend 
     * the produced the result.
     *
     * @return the score
     */
    BigDecimal getScore();
}
