/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.mimetypes.image;

import com.arsdigita.mimetypes.ImageMimeType;
import com.arsdigita.mimetypes.MimeTypeCollection;

import java.util.HashMap;

import org.apache.log4j.Logger;

/**
 * Implements the Factory pattern in order to
 * supply implementation-specific instances of {@link ImageSizer}.
 * The factory is initialized in the mime-type initializer.
 *
 * @author <a href="mailto:sfreidin@arsdigita.com">Stanislav Freidin</a>
 * @version $Id: //core-platform/dev/src/com/arsdigita/mimetypes/image/ImageSizerFactory.java#4 $
 */
public class ImageSizerFactory {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/mimetypes/image/ImageSizerFactory.java#4 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static HashMap s_sizers = new HashMap();

    private static final Logger s_log = Logger.getLogger(ImageSizerFactory.class);
    /**
     * Constructor
     */
    public ImageSizerFactory() {}

    /**
     * Add an {@link ImageSizer} instance to the factory
     *
     * @param mime the mime-type of the image which the image sizer
     *   is able to handle
     * @param sizer an implementation-specific instance of {@link ImageSizer}
     */
    public static void addImageSizer(String mime, ImageSizer sizer) {
        s_sizers.put(mime, sizer);
    }

    /**
     * Obtain an instance of an {@link ImageSizer} which can determine
     * the size of an image with the given mime-type.
     *
     * @param mime the mime-type of the image which the image sizer
     *   should handle
     * @return an appropriate instance of {@link ImageSizer}, or null
     *   if no such instance exists
     */
    public static ImageSizer getImageSizer(String mime) {
        return (ImageSizer)s_sizers.get(mime);
    }

    /**
     * Initialize the factory by loading all image sizers from the
     * database
     */
    public static void initialize() {
        MimeTypeCollection mimes = ImageMimeType.getAllImageMimeTypes();
        while(mimes.next()) {
            ImageMimeType mime = (ImageMimeType)mimes.getMimeType();
            String sizerName = mime.getImageSizer();
            if(sizerName != null) {
                try {
                    Class sizerClass = Class.forName(sizerName);
                    ImageSizer sizer = (ImageSizer)sizerClass.newInstance();
                    addImageSizer(mime.getMimeType(), sizer);
                } catch (Exception e) {
                    s_log.error("Initialization error", e);
                    // Do nothing
                }
            }
        }
    }

}
