/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel.security;

import java.math.BigDecimal;
import java.math.BigInteger;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginException;

/**
 * Provide access to the JSESSIONID value as a BigDecimal.  This class
 * used to secure the JSESSIONID using a cryptographic hash, but we
 * have decided to rely on the web container to secure the JSESSIONID
 * because the web container is responsible for setting JSESSIONID
 * anyway.  We cannot simply remove this class because
 * KernelRequestContext has a getSessionContext method and is "Stable".
 *
 * @author Rob Mayoff
 **/
public class SessionContext {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/kernel/security/SessionContext.java#10 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log =
        Logger.getLogger(SessionContext.class.getName());

    private HttpServletRequest m_req;
    private BigDecimal m_sessionID;

    /**
     * Creates a session context from an HTTP request.  Package-private.
     * Code should access this class via
     * KernelHelper.getKernelRequestContext(req).getSessionContext().
     **/
    SessionContext(HttpServletRequest req,
                   HttpServletResponse res) {
        m_req = req;
    }

    /**
     * Returns the current session ID.
     *
     * @return the session ID.
     **/
    public BigDecimal getSessionID() {
        if (m_sessionID == null)
            throw new IllegalStateException("session ID is not defined");
        return m_sessionID;
    }

    /**
     * Converts the session ID to a BigDecimal for backward
     * compatibility.
     * Package-private.
     *
     * @throws LoginException if unable to load session ID.
     **/
    void loadSessionID(CallbackHandler handler) throws LoginException {
        s_log.debug("START loadSessionID()");
        try {
            m_sessionID = new BigDecimal
                (new BigInteger(1, Util.getRequest(handler)
                                .getSession().getId().getBytes(Crypto.CHARACTER_ENCODING)));
        } catch (java.io.UnsupportedEncodingException e) {
            throw new KernelLoginException("Could not get sessionID: ", e);
        }
        s_log.debug("sessionID = " + m_sessionID);
    }
}
