/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel.permissions;


import com.arsdigita.domain.DomainQuery;

import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.OID;

import com.arsdigita.kernel.User;
import com.arsdigita.kernel.Group;
import com.arsdigita.kernel.PersonName;
import com.arsdigita.kernel.EmailAddress;

import java.math.BigDecimal;

/**
 * 
 * Represents a collection of permissions that have been
 * granted on a single object.
 *
 * @see PermissionService
 * @see PermissionDescriptor
 * @see PrivilegeDescriptor
 * @see com.arsdigita.kernel.User
 * @see com.arsdigita.kernel.Group
 *
 * @author Oumi Mehrotra 
 * @version 1.0
 **/
public class ObjectPermissionCollection extends DomainQuery {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/kernel/permissions/ObjectPermissionCollection.java#10 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Constructor.
     *
     * @see com.arsdigita.domain.DomainCollection#DomainCollection(DataCollection)
     **/
    protected ObjectPermissionCollection(DataQuery query) {
        super(query);
    }

    /**
     * Gets the party ID of the grantee of the current permission record.
     * @return the party ID of the grantee of the current permission record.
     */
    public BigDecimal getGranteeID() {
        return (BigDecimal) m_dataQuery.get("granteeID");
    }

    /**
     * Returns the party OID of the grantee of the current permission record.
     * The returned OID's object type is a subtype of
     * Party.BASE_DATA_OBJECT_TYPE.
     * @return the party OID of the grantee of the current permission record.
     */
    public OID getGranteeOID() {
        if (granteeIsUser()) {
            return new OID (User.BASE_DATA_OBJECT_TYPE,
                            m_dataQuery.get("granteeID"));
        } else {
            return new OID (Group.BASE_DATA_OBJECT_TYPE,
                            m_dataQuery.get("granteeID"));
        }
    }

    /**
     * Determines whether the grantee of the current permission record
     * is a User (as opposed to a Group).
     * @return <code>true</code> if the grantee is a User; <code>false</code> if
     * the grantee is a  Group.
     *
     * @see com.arsdigita.kernel.User
     * @see com.arsdigita.kernel.Group
     */
    public boolean granteeIsUser() {
        return ((Boolean) m_dataQuery.get("granteeIsUser")).booleanValue();
    }

    /**
     * Returns the name of the grantee of the current permission record.
     * @return the name of the grantee of the current permission record.
     *
     * @see com.arsdigita.kernel.Party#getName()
     */
    public String getGranteeName() {
        if (!granteeIsUser()) {
            return (String) m_dataQuery.get("granteeName");
        }
        return getGranteePersonName().toString();
    }

    /**
     * Returns a PersonName object representing the name of the grantee of
     * the current permission record, assuming the grantee is a User.
     * @return  a PersonName object representing the name of the grantee of
     * the current permission record.
     *
     * @see com.arsdigita.kernel.User#getPersonName()
     */
    public PersonName getGranteePersonName() {
        if (!granteeIsUser()) {
            throw new RuntimeException ("Grantee is a group, not a user.");
        }

        return new PersonName((DataObject) get("granteePersonName"));
    }

    /**
     * Gets the primary email address of the grantee of
     * the current permission record.
     * @return the primary email address of the grantee of
     * the current permission record.
     *
     * @see com.arsdigita.kernel.Party#getPrimaryEmail()
     */
    public EmailAddress getGranteeEmail() {
        return new EmailAddress((String) m_dataQuery.get("granteeEmail"));
    }

    /**
     * Gets the granted privilege.
     * @return the granted privilege.
     */
    public PrivilegeDescriptor getPrivilege() {
        return PrivilegeDescriptor.get((String) m_dataQuery.get("privilege"));
    }

    /**
     * Determines whether the current permission record was inherited from the
     * permission context of the object of this permission collection.
     *
     * @return <code>true</code> if the current permission record was inherited from the
     * permission context of the object of this permission collection; <code>false</code>
     * otherwise.
     */
    public boolean isInherited() {
        return ((Boolean) m_dataQuery.get("isInherited")).booleanValue();
    }
}
