/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.domain.DomainObjectFactory;
import com.arsdigita.domain.DomainObjectInstantiator;
import com.arsdigita.domain.GlobalObserverManager;
import com.arsdigita.initializer.Configuration;
import com.arsdigita.initializer.InitializationException;
import com.arsdigita.kernel.PackageInstance;
import com.arsdigita.kernel.PackageType;
import com.arsdigita.kernel.SiteNode;
import com.arsdigita.kernel.Stylesheet;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PermissionsObserver;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.kernel.permissions.UniversalPermissionDescriptor;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import com.arsdigita.util.Assert;

import java.util.Locale;

import org.apache.log4j.Logger;

/**
 * Initializes the Kernel and bootstraps the rest of the system.
 *
 * @version $Revision: #38 $ $Date: 2004/04/07 $
 */
public class Initializer extends BaseInitializer {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/kernel/Initializer.java#38 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger(Initializer.class);

    private Configuration m_conf = new Configuration();

    public Configuration getConfiguration() {
        return m_conf;
    }

    protected void doStartup() {
        setupDomainFactory();
        setupURLService();

        TransactionContext txn = SessionManager.getSession()
            .getTransactionContext();
        txn.beginTxn();

        // Initialize privilege descriptors used in permissions service
        s_log.debug("Initializing privilege descriptors...");
        PrivilegeDescriptor.initialize();
        s_log.debug("Done.");

        txn.commitTxn();
    }

    private void setupDomainFactory() {
        DomainObjectInstantiator instantiator;

        /*** ACSObject ***/
        // register instantiator for ACSObject data object type
        instantiator = new ACSObjectInstantiator();
        DomainObjectFactory.registerInstantiator(ACSObject.BASE_DATA_OBJECT_TYPE,
                                                 instantiator);

        /*** Party ***/
        // We use the same instantiator as for ACSObject because party is
        // abstract so we don't need to override doNewInstance().
        DomainObjectFactory.registerInstantiator(Party.BASE_DATA_OBJECT_TYPE,
                                                 instantiator);

        /*** User ***/
        instantiator = new ACSObjectInstantiator() {
                public DomainObject doNewInstance(DataObject dataObject) {
                    return new User(dataObject);
                }
            };
        DomainObjectFactory.registerInstantiator
            (User.BASE_DATA_OBJECT_TYPE, instantiator);

        /*** Group ***/
        instantiator = new ACSObjectInstantiator() {
                public DomainObject doNewInstance(DataObject dataObject) {
                    return new Group(dataObject);
                }
            };
        DomainObjectFactory.registerInstantiator(Group.BASE_DATA_OBJECT_TYPE,
                                                 instantiator);
    }

    /* Register URLFinders with the URLService */
    private void setupURLService() {
        // PackageInstance is the only kernel object type for which kernel
        // can provide a URLFinder.  Other object types could have
        // finders registered for them by other initializers (in UI packages).
        // For PackageInstance, urls are determined from the mount points on
        // the site map.
        URLService.registerFinder(PackageInstance.BASE_DATA_OBJECT_TYPE,
                                  new GenericURLFinder(""));
    }

    protected void doShutdown() {
        // Empty
    }
}
