/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.formbuilder.parameters;


// The DomainObject class that we are extending
import com.arsdigita.kernel.ACSObject;

// For the id of the persistent parameter listener
import java.math.BigDecimal;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.DataObject;

// Thrown if the underlying DataObject with given id cannot be found
import com.arsdigita.domain.DataObjectNotFoundException;

// The interface of the classes we are persisting
import com.arsdigita.bebop.event.ParameterListener;

// To instantiate listener of right class
import com.arsdigita.formbuilder.util.FormBuilderUtil;

// We work with these classes to store attributes of listeners
import com.arsdigita.bebop.util.Attributes;
import com.arsdigita.formbuilder.util.AttributeHelper;

// ACS 5 uses Log4J for logging
import org.apache.log4j.Logger;

import com.arsdigita.util.Assert;


/**
 * The base class of all persistent parameter listeners.
 *
 * @author Peter Marklund
 * @version $Id: //core-platform/dev/src/com/arsdigita/formbuilder/parameters/PersistentParameterListener.java#11 $
 *
 */
public class PersistentParameterListener extends ACSObject {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/formbuilder/parameters/PersistentParameterListener.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log =
        Logger.getLogger(PersistentParameterListener.class.getName());

    // Attributes are stored in memory in this object before they
    // are saved in the attributeString attribute
    private Attributes m_attributes;
    private boolean m_attributeChanged = false;

    /**
     * The fully qualified name of the underlying DataObject of this class.
     */
    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.formbuilder.Listener";

    // *** Constructors -------------

    /**
     * Constructor for creating a new listener that can be persisted
     */
    public PersistentParameterListener() {
        super(BASE_DATA_OBJECT_TYPE);
        set("defaultDomainClass", this.getClass().getName());
    }


    /**
     * Constructor for creating a new listener that can be persisted
     */
    public PersistentParameterListener(String className) {

        super(BASE_DATA_OBJECT_TYPE);

        set("className", className);

        set("defaultDomainClass", this.getClass().getName());
    }


    /**
     * Constructor for retrieving a persistent parameter listener
     * from the database
     */
    public PersistentParameterListener(BigDecimal id)
        throws DataObjectNotFoundException {

        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    /**
     * Internal constructor to be used by subclasses
     */
    protected PersistentParameterListener(OID oID)
        throws DataObjectNotFoundException {

        super(oID);
    }

    public PersistentParameterListener(DataObject obj) {
        super(obj);
    }

    protected void beforeSave() {
        if (m_attributeChanged) {
            Assert.assertNotNull(m_attributes, "Attribute map");
            set("attributeString",
                AttributeHelper.getAttributeString(m_attributes));
            m_attributeChanged = false;
        }

        super.beforeSave();
    }

    /**
     * Factory method that creates the Bebop component whose
     * persistence is handled by this domain object.
     */
    public ParameterListener createListener() {

        return (ParameterListener)FormBuilderUtil.instantiateObject(getClassName());
    }

    // *** Attribute Methods

    public void setClassName(String className) {
        set("className", className);
    }

    public String getClassName() {
        return (String)get("className");
    }

    /**
     * For use internally by extending classes to set Listener attributes
     */
    protected void setListenerAttribute(String name, String value) {
        if (m_attributes == null)
            loadAttributes();

        m_attributes.setAttribute(name, value);
        m_attributeChanged = true;
        if (m_attributeChanged == false) {
            m_attributeChanged = true;
            // force persistence to run beforeSave eventually
            set("attributeString", get("attributeString"));
        }
    }

    /**
     * For use internally by extending classes to get Listener attributes
     */
    protected String getListenerAttribute(String name) {
        if (m_attributes == null)
            loadAttributes();

        return m_attributes.getAttribute(name);
    }

    private void loadAttributes() {
        String attrs = (String)get("attributeString");
        if (attrs == null) {
            m_attributes = new Attributes();
        } else {
            m_attributes = (new AttributeHelper()).getAttributesMap((String)get("attributeString"));
        }
        m_attributeChanged = false;
    }
}
