/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.domain;

import com.arsdigita.persistence.PersistenceException;
import com.arsdigita.persistence.DataObject;

/**
 * Interface for DomainObject observers. Implementations of
 * DomainObjectObserver should implement the <code>equals</code> method
 * meaningfully so that multiple copies of the same observer are not added by
 * multiple domain objects that wrap the same data object.
 *
 * @author Joseph Bank
 * @version 1.0
 **/
public interface DomainObjectObserver {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/domain/DomainObjectObserver.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Callback for a set operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     * @param name the name of the parameter being set
     * @param old_value the old value of the parameter being set
     * @param new_value the old value of the parameter being set
     */
    public void set(DomainObject dobj,
                    String name,
                    Object old_value,
                    Object new_value);

    /**
     * Callback for an add operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     * @param name the name of the parameter being set
     * @param dobj the object added
     */
    public void add(DomainObject dobj,
                    String name, DataObject dataObject);
    /**
     * Callback for a remove operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     * @param name the name of the parameter being set
     * @param dobj the object removed
     */
    public void remove(DomainObject dobj,
                       String name, DataObject dataObject);
    /**
     * Callback for a clear operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     * @param name the name of the parameter being set
     */
    public void clear(DomainObject dobj, String name);

    /**
     * Callback before a save operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     */
    public void beforeSave(DomainObject dobj) throws PersistenceException;

    /**
     * Callback after a save operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     */
    public void afterSave(DomainObject dobj) throws PersistenceException;

    /**
     * Callback before a delete operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     */
    public void beforeDelete(DomainObject dobj) throws PersistenceException;

    /**
     * Callback after a delete operation on the observed DomainObject
     *
     * @param dobj the observed domain object
     */
    public void afterDelete(DomainObject dobj) throws PersistenceException;

}
