/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.categorization;

import com.arsdigita.kernel.ACSObjectCollection;
import com.arsdigita.persistence.DataCollection;

/**
 * Represents a collection of categorized objects.
 *
 * <p>This collection can be iterated over by calling the {@link #next() next()}
 * method and retrieving the current object via {@link #getDomainObject()
 * getDomainObject()}.</p>
 *
 * <p>This class inherits a number of methods from {@link
 * com.arsdigita.domain.DomainQuery} that allow filtering and ordering of this
 * collection.  In order to apply filters or order clauses, you have to know the
 * object model of the {@link DataCollection data collection} backing up this
 * class. Categorized collections are produced by {@link
 * Category#getObjects(String)} and {@link Category#getObjects(String, String)}.
 * See the Javadoc for these methods to learn about high-level guarantees of the
 * structure of the possible query paths that can be used for ordering and/or
 * filtering. </p>
 *
 * @see Category#getObjects(String)
 * @see Category#getObjects(String, String)
 *
 * @author  Vadim Nasardinov (vadimn@redhat.com)
 * @since   2003-09-17
 * @version $Revision: #6 $ $DateTime: 2004/04/07 16:07:11 $
 **/
public final class CategorizedCollection extends ACSObjectCollection {

    private String m_sortPath = null;

    CategorizedCollection(DataCollection dataCollection) {
        this(dataCollection, null);
    }

    CategorizedCollection(DataCollection dataCollection, String sortPath) {
        super(dataCollection);
        m_sortPath = sortPath;
    }

    /**
     * Specifies the sort order the categorized collection.
     *
     * <p>A categorization collection usually represents a list of child objects
     * of a particular category.  These objects can be sorted using the {@link
     * Category#swapWithNext(ACSObject)} and {@link
     * Category#swapWithPrevious(ACSObject)} methods.  This method to make use
     * of this sort order.</p>
     *
     * @see Category#SORT_KEY
     * @see Category#swapWithNext(ACSObject)
     * @see Category#swapWithPrevious(ACSObject)
     * @see Category#alphabetizeChildCategories()
     **/
    public final void sort(boolean ascending) {
        if (m_sortPath != null) {
            if ( ascending ) {
                addOrder(m_sortPath + " asc");
            } else {
                addOrder(m_sortPath + " desc");
            }
        }
    }
}
