/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.util;

import com.arsdigita.util.Assert;
import com.arsdigita.util.Lockable;

import java.util.Map;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Collection;

/**
 * This class represents a set of key-value pairs, for use in
 * extending the XML attributes of Bebop components.
 *
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/util/Attributes.java#9 $
 */
public class Attributes implements Lockable, Cloneable {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/util/Attributes.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    /**
     * Map of attributes.
     */
    private HashMap m_attributes;

    private boolean m_locked;

    /**
     * Creates an Attributes object.
     */
    public Attributes() {
        m_attributes = new HashMap();
        m_locked = false;
    }

    /**
     * Clone the attributes. The clone is not locked and has its own set of
     * attributes and values.
     * @post ! ((Attributes) return).isLocked()
     */
    public Object clone() throws CloneNotSupportedException {
        Attributes result = (Attributes) super.clone();
        result.m_attributes = (HashMap) m_attributes.clone();
        result.m_locked = false;
        return result;
    }

    /**
     * <p>Sets an arbitrary attribute for inclusion in the HTML tags that
     * compose element.  For standard attributes in the HTML 4.0
     * specification, use of this method has the same effect as the
     * specific mutator method provided for each attribute.</p>
     *
     * <p>Setting an attribute <code>name</code> to <code>null</code>
     * removes it.</p>
     *
     * @param name The name of the attribute
     * @param value The value to assign the named attribute
     */
    public void setAttribute(String name, String value) {
        Assert.assertNotLocked(this);
        name = name.toLowerCase();
        m_attributes.put(name, value);
    }

    /**
     * Return the value of an attribute.
     *
     * @pre name != null
     *
     * @param name the name of the attribute
     * @return the value set previously with
     * {@link #setAttribute setAttribute}
     */
    public String getAttribute(String name) {
        return (String) m_attributes.get(name.toLowerCase());
    }

    /**
     *  Return a collection of all of the attribute keys represented.
     *  This, along with {@link #getAttribute(String name)} allows
     *  you to iterate through all of the attributes.  All elements
     *  of the Collection are Strings
     */
    public Collection getAttributeKeys() {
        return m_attributes.keySet();
    }


    /**
     * Copy all attributes into the given DOM Element.  This will
     * override any preexisting Element attributes of the same names.
     */
    public void exportAttributes(com.arsdigita.xml.Element target) {
        Iterator attributesIterator = m_attributes.entrySet().iterator();

        while (attributesIterator.hasNext()) {
            Map.Entry entry = (Map.Entry) attributesIterator.next();

            if (entry.getValue() != null) {
                target.addAttribute((String) entry.getKey(),
                                    (String) entry.getValue());
            }
        }
    }

    public void lock() {
        m_locked = true;
    }

    public final boolean isLocked() {
        return m_locked;
    }
}
