/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.form;


import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringParameter;
// This interface contains the XML element name of this class
// in a constant which is used when generating XML
import com.arsdigita.bebop.util.BebopConstants;


import java.util.Iterator;

import com.arsdigita.xml.Element;

/**
 *     A class
 *    representing a <em>group</em> of associated radio buttons.
 *
 *    @author Karl Goldstein 
 *    @author Uday Mathur 
 *    @author Rory Solomon 
 *    @author Michael Pih 
 *    @version $Id: //core-platform/dev/src/com/arsdigita/bebop/form/RadioGroup.java#9 $ */
public class RadioGroup extends OptionGroup implements BebopConstants {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/form/RadioGroup.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    // xml attribute for layout
    private final static String AXIS = "axis";

    /**
     * Specifies that options should be laid out left to right.
     */
    // this is default
    public final static int HORIZONTAL = 1;

    /**
     * Specifies that options should be laid out top to bottom.
     */
    public final static int VERTICAL = 2;

    public RadioGroup(String name) {
        this(new StringParameter(name));
    }

    public RadioGroup(ParameterModel model) {
        super(model);
        m_xmlElement = BEBOP_RADIO;
    }

    /**
     *  Returns a string naming the type of this widget.
     */
    public String getType() {
        return "radio";
    }

    /** The XML tag.
     *  @return The tag to be used for the top level DOM element
     *  generated for this type of Widget.  */
    protected String getElementTag() {
        return BEBOP_RADIOGROUP;
    }

    /**
     * Generates the DOM for a radio button group
     * <p>Generates DOM fragment:
     * <p><pre><code>&lt;bebop:radioGroup name=...>
     *   &lt;bebop:radio name=... value=... label=... [checked=...]>
     *     option value
     *   &lt;/bebop:option>
     * ...
     * &lt;/bebop:radioGroup></code></pre>
     */
    public void generateWidget(PageState state, Element parent) {
        Element radioGroup = parent.newChildElement (getElementTag(), BEBOP_XML_NS);
        exportAttributes(radioGroup);

        for ( Iterator i = getOptions(state); i.hasNext(); ) {
            Option o = (Option)i.next();
            o.generateXML(state, radioGroup);
        }
    }

    /**
     * Is this a multiple (and not single) selection option group?
     *
     * @return true if this OptionGroup can have more than one
     * selected option; false otherwise.
     */
    public boolean isMultiple() {
        return false;
    }

    /**
     * Sets the layout for the options in this radio group.
     *
     * @param layout one of RadioGroup.VERTICAL or RadioGroup.HORIZONTAL
     **/
    public void setLayout(int layout) {
        setAttribute(AXIS, String.valueOf(layout));
    }

    /**
     * Returns the layout for the options in this radio group.
     *
     * @return one of RadioGroup.VERTICAL or RadioGroup.HORIZONTAL
     **/
    public int getLayout() {
        String value = getAttribute(AXIS);
        if (value == null) {
            return HORIZONTAL;
        } else if (value.equals(String.valueOf(HORIZONTAL))) {
            return HORIZONTAL;
        } else if (value.equals(String.valueOf(VERTICAL))) {
            return VERTICAL;
        } else {
            throw new IllegalStateException(
                                            "invalid value for axis attribute: " + value);
        }
    }
}
