/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel.security;

import java.security.*;
import java.util.*;
import javax.crypto.Mac;

import junit.framework.*;

public class CredentialTest extends TestCase {

    private static SecureRandom s_random = new SecureRandom();

    public CredentialTest(String name) {
        super(name);
    }
    public static Test suite() {
        try {
            return new TestSuite(CredentialTest.class);
        } catch (final Throwable t) {
            // handles NoClassDefFoundError
            // and ExceptionInInitializerError
            return new TestCase("Create CredentialTest") {
                    public void runTest() throws Throwable {
                        throw t;
                    }
                };
        }
    }
    public void testCycle() {
        // create -> toString -> parse -> toString -> compare
        Credential c1 = null, c2 = null;
        try {
            c1 = Credential.create
                ("12345678901234567890", 1000*60);
        } catch (CredentialException e) {
            fail("Credential.create: "+e);
        }
        String s1 = c1.toString();
        try {
            c2 = Credential.parse(s1);
        } catch (CredentialException e) {
            fail("Credential.parse: "+e);
        }
        String s2 = c2.toString();
        // strings equal?
        assertEquals(s1, s2);
    }

    public void testExpired() {
        Credential c1 = null;
        try {
            c1 = Credential.create
                ("12345678901234567890", -1000*60);
            fail("Expected CredentialEncodingException");
        } catch (CredentialEncodingException e) {
            //success
        }
    }

    public void testInvalidValue() {
        Credential c1 = null;
        try {
            c1 = Credential.create
                ("12345678901234567890"+Credential.SEPARATOR,
                 1000*60);
            fail("Expected CredentialEncodingException");
        } catch (CredentialEncodingException e) {
            // success
        } catch (CredentialException e) {
            fail("Expected CredentialEncodingException, but got: "+e);
        }
    }

    public void testInvalidKey() throws GeneralSecurityException {
        byte[] key1 = Store.newKey();
        byte[] key2 = new byte[key1.length];
        System.arraycopy(key1, 0, key2, 0, key1.length);
        key2[0] = (byte) ~key2[0];

        Credential c1 = Credential.create
            ("12345678901234567890", 1000*60, Crypto.newMac(key1));
        String s = c1.toString();

        // use the wrong key
        Mac mac = Crypto.newMac(key2);

        try {
            Credential.parse(s, mac);
            fail("Expected CredentialParsingException");
        } catch (CredentialParsingException e) {
            ; // success
        } catch (CredentialException e) {
            fail("Expected CredentialParsingException, but got: "+e);
        }
    }

    public void testInvalidString() {
        Credential c1 = null, c2 = null;
        try {
            c1 = Credential.create
                ("12345678901234567890", 1000*60);
        } catch (CredentialException e) {
            fail("Credential.create: "+e);
        }
        String s1 = c1.toString();
        // use the wrong string
        String s2 = "invalid" + s1;
        try {
            c2 = Credential.parse(s2);
            fail("Expected CredentialParsingException");
        } catch (CredentialParsingException e) {
            // success
        } catch (CredentialException e) {
            fail("Expected CredentialParsingException, but got: "+e);
        }
    }
}
