/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.web;

import java.util.ArrayList;
import java.util.Iterator;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;

/**
 * <p>A class that provides request-framed control over a thread-local
 * value.  With such control, it is possible to safely reuse
 * thread-local data across requests.  For example, the following
 * <code>InternalRequestLocal</code> reuses a
 * <code>HashMap</code>.</p>
 *
 * <pre><blockquote>
 * class HashMapRequestLocal extends InternalRequestLocal {
 *     protected Object initialValue() {
 *         return new HashMap();
 *     }
 *
 *     // Does not override prepareValue(HttpServletRequest).
 *     // InternelRequestLocal's default implementation calls
 *     // clearValue() to prepare the value for the request.
 *
 *     protected void clearValue() {
 *         ((HashMap) get()).clear();
 *     }
 * }
 * </blockquote></pre>
 *
 * <p><code>initialValue()</code> is called just once, when the value
 * is first accessed.  <code>prepareValue(HttpServletRequest)</code>
 * is called at the start of every request serviced by {@link
 * com.arsdigita.web.BaseServlet}.  <code>clearValue()</code> is
 * called at the end of every request handled by said servlet.</p>
 *
 * <p>The default implementation of <code>clearValue()</code> sets the
 * value to <code>null</code>, and the default implementation of
 * <code>prepareValue(HttpServletRequest)</code> calls
 * <code>clearValue()</code>.  As a result, an
 * <code>InternalRequestLocal</code> as used in the following example
 * is similar to using a request attribute.</p>
 *
 * <pre><blockquote>
 * // This value is s_servletContext.set(null) at the start and end of
 * // each request.
 * static ThreadLocal s_servletContext = new InternalRequestLocal();
 * </blockquote></pre>
 *
 * <p>Be advised that errors in using this class can easily result in
 * excess trash left on threads and, worse, big memory leaks.  Please
 * use caution.</p>
 *
 * @see java.lang.ThreadLocal
 * @see com.arsdigita.web.BaseServlet
 * @author Justin Ross &lt;<a href="mailto:jross@redhat.com">jross@redhat.com</a>&gt;
 * @version $Id: //core-platform/dev/src/com/arsdigita/web/InternalRequestLocal.java#7 $
 */
class InternalRequestLocal extends ThreadLocal {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/web/InternalRequestLocal.java#7 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (InternalRequestLocal.class);

    private static final ArrayList s_locals = new ArrayList();

    static final void prepareAll(final HttpServletRequest sreq) {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Initializing all request-local objects; there are " +
                        s_locals.size());
        }

        final Iterator iter = s_locals.iterator();

        while (iter.hasNext()) {
            final InternalRequestLocal local = (InternalRequestLocal) iter.next();

            local.prepareValue(sreq);
        }
    }

    static final void clearAll() {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Clearing all request-local objects; there are " +
                        s_locals.size());
        }

        final Iterator iter = s_locals.iterator();

        while (iter.hasNext()) {
            final InternalRequestLocal local = (InternalRequestLocal) iter.next();

            local.clearValue();
        }
    }

    /**
     * <p>Constructs a new InternalRequestLocal and registers it to be
     * initialized and cleared on each request.</p>
     */
    public InternalRequestLocal() {
        super();

        s_locals.add(this);
    }

    /**
     * <p>Called at the start of each request, this method returns the
     * request-initialized value of the thread-local variable.</p>
     *
     * <p>By default this method calls <code>clearValue()</code>.</p>
     *
     * @param sreq the current servlet request
     * @return the request-initialized value
     */
    protected void prepareValue(HttpServletRequest sreq) {
        clearValue();
    }

    /**
     * <p>Called at the end of each request, this method clears the
     * thread-local value.</p>
     *
     * <p>By default this method calls <code>set(null)</code>.  Users
     * of this class may override this method to better reuse the
     * thread-local value.</p>
     */
    protected void clearValue() {
        set(null);
    }
}
