/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.versioning;

import com.arsdigita.util.Assert;

// new versioning

/**
 * Type-safe enum providing constants for marking four types of nodes in the
 * versioning dependence graph.
 *
 * @author Vadim Nasardinov (vadimn@redhat.com)
 * @since 2003-03-03
 * @version $Revision: #5 $ $DateTime: 2004/04/07 16:07:11 $
 **/
final class NodeType {
    private String m_type;
    private int m_order;

    /**
     * Labels object types whose been PDL definition has been marked "versioned"
     * or subtypes of such object types.
     **/
    public final static NodeType VERSIONED_TYPE   =
        new NodeType("versioned type", 3);

    /**
     * Labels an object type that is not explicitly marked versioned (and whose
     * supertype is not explicitly marked versioned), but that is nonetheless
     * fully versioned, because it is a component of a versioned type.
     **/
    public final static NodeType COVERSIONED_TYPE =
        new NodeType("coversioned type", 2);

    /**
     * Labels an object type that is not fully versioned, but for which we
     * record enough information for undeleting a deleted instance.
     **/
    public final static NodeType RECOVERABLE      =
        new NodeType("recoverable", 1);

    /**
     * Labels an object type that is ignored by the versioning system. (It is
     * called unreachable, because it can be reached from versioned and/or
     * recoverable nodes in the versioning graph.)
     **/
    public final static NodeType UNREACHABLE      =
        new NodeType("unreachable", 0);

    private NodeType(String type, int order) {
        m_type = type;
        m_order = order;
    }

    public String toString() {
        return m_type;
    }

    /**
     * Defines a complete order over this enum.
     *
     * Ascending enumeration: {@link #UNREACHABLE}, {@link #RECOVERABLE}, {@link
     * #COVERSIONED_TYPE}, {@link #VERSIONED_TYPE}.
     **/
    public boolean lessThan(NodeType type) {
        Assert.exists(type, NodeType.class);
        return m_order < type.m_order;
    }
}
