/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.ui.admin;

import com.arsdigita.bebop.BebopMapDispatcher;
import com.arsdigita.bebop.Page;
import com.arsdigita.dispatcher.DispatcherHelper;
import com.arsdigita.dispatcher.RequestContext;
import com.arsdigita.web.RedirectSignal;
import com.arsdigita.web.ParameterMap;
import com.arsdigita.web.URL;
import com.arsdigita.web.Web;
import com.arsdigita.kernel.SiteNode;
import com.arsdigita.kernel.Party;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.permissions.PermissionDescriptor;
import com.arsdigita.kernel.permissions.PermissionService;
import com.arsdigita.kernel.permissions.PrivilegeDescriptor;
import com.arsdigita.kernel.security.Initializer;
import java.io.IOException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import org.apache.log4j.Logger;

/**
 * This is the base dispatcher extended by the Admin, User and Group
 * dispatchers.  It defines a dispatch method that verifies that the
 * user requesting the page is logged in and has site-wide
 * administration privileges.
 *
 * @author Ron Henderson
 * @version $Id: //core-platform/dev/src/com/arsdigita/ui/admin/SubsiteDispatcher.java#12 $
 */
class SubsiteDispatcher extends BebopMapDispatcher {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/ui/admin/SubsiteDispatcher.java#12 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (SubsiteDispatcher.class);

    /**
     * Adds one URL the dispatcher's page map.  If isIndex is true,
     * the page is also mapped to the empty string "" so that it can
     * handle the special case of a request for the directory.
     */
    public void addPage(String url, Page p, boolean isIndex) {
        if (isIndex) {
            super.addPage("", p);
        }

        super.addPage(url, p);
    }

    /**
     * Examines each request for a page in the admin section to verify
     * that the user requesting the page is logged in and authorized
     * as a system-wide administrator.
     */
    public void dispatch(HttpServletRequest req,
                         HttpServletResponse resp,
                         RequestContext ctx)
            throws IOException, javax.servlet.ServletException {
        // Always dispatch the "access denied" page

        String requestURI = req.getRequestURI();

        if (requestURI.endsWith("/denied/")) {
            super.dispatch(req, resp, ctx);
            return;
        }

        Party party = Kernel.getContext().getParty();

        if (party == null) {
            // The user is not logged in; redirect to the login page.

            final String path = Initializer.getSecurityHelper().getLoginURL
                (req);

            final ParameterMap params = new ParameterMap();

            params.setParameter("return_url",
                                Web.getContext().getRequestURL());

            throw new RedirectSignal(URL.there(req, path, params), false);
        } else {
            // Check if user has an admin privilege.

            String processedURL = ctx.getProcessedURLPart();

            SiteNode node;

            node = SiteNode.getSiteNode(processedURL);

            PermissionDescriptor admin = new PermissionDescriptor
                (PrivilegeDescriptor.ADMIN, node, party);

            // Admins should always see the very latest data
            DispatcherHelper.cacheDisable(resp);

            if (PermissionService.checkPermission(admin)) {
                super.dispatch(req, resp, ctx);
            } else {
                final URL url = URL.there(req, node.getURL() + "denied");

                throw new RedirectSignal(url, false);
            }
        }
    }
}
