/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.toolbox.ui;

import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.SimpleContainer;
import com.arsdigita.bebop.PageState;
import com.arsdigita.kernel.Kernel;
import com.arsdigita.kernel.Party;
import com.arsdigita.xml.Element;


/**
 * <p>A <code>SecurityContainer</code> adds an access check to a
 * {@link com.arsdigita.bebop.Component}. The child component is made
 * invisible if the current user cannot access the it.</p>
 *
 * <p><b>Warning:</b> - A call to <code>setVisible(state, true)</code> does
 * not necessarily mean that <code>isVisible(state)</code> will return
 * true, since the <code>isVisible</code> also takes security checks
 * into account.</p>
 *
 * <p>General usage of the <code>SecurityContainer</code> is as follows:</p>
 *
 * <blockquote><code><pre>
 * MyComponent c = new MyComponent();
 * SecurityContainer sc = new SecurityContainer(c) {
 *   protected boolean canAccess(User user, PageState state) {
 *     return ( user != null );
 *   }
 * };
 * add(sc);
 * </pre></code></blockquote>
 *
 * @author Michael Pih 
 * @version $Revision: #9 $ $DateTime: 2004/04/07 16:07:11 $
 */
public abstract class SecurityContainer extends SimpleContainer {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/toolbox/ui/SecurityContainer.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * This default constructor should be followed by calls to
     * <code>add</code>.
     * */
    public SecurityContainer() {}

    /**
     * Create a <code>SecurityContainer</code> around a child component.
     *
     * @param c The child component
     */
    public SecurityContainer(Component c) {
        add(c);
    }

    /**
     * Is the component visible?
     *
     * @param state The page state
     * @return true if the component is visible, false otherwise
     */
    public boolean isVisible(PageState state) {
        Party party = Kernel.getContext().getParty();
        return ( super.isVisible(state) && canAccess(party, state) );
    }

    /**
     * Returns true if the current user can access the child component.
     *
     * @param party The party
     * @param state The page state
     * @return true if the access checks pass, false otherwise
     */
    protected abstract boolean canAccess(Party party, PageState state);

    /**
     * Generates the XML for the child component if this
     * component is visible.
     *
     * @param state The page state
     * @param parent The parent DOM element
     */
    public void generateXML(PageState state, Element parent) {
        if ( isVisible(state) ) {
            super.generateXML(state, parent);
        }
    }

}
