/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.search.lucene;


import com.arsdigita.search.QueryEngine;
import com.arsdigita.search.Search;
import com.arsdigita.search.ResultSet;
import com.arsdigita.search.QuerySpecification;
import com.arsdigita.search.FilterSpecification;
import com.arsdigita.search.FilterType;

import com.arsdigita.search.filters.ObjectTypeFilterSpecification;
import com.arsdigita.search.filters.ObjectTypeFilterType;

import com.arsdigita.persistence.metadata.ObjectType;

import com.arsdigita.util.StringUtils;

import java.util.List;
import java.util.ArrayList;

import java.math.BigDecimal;

import org.apache.lucene.search.Filter;


/**
 * This provides the basic lucene query engine implementation
 * which can restrict based on object type
 * @see com.arsdigita.search.QueryEngine
 */
public class BaseQueryEngine implements QueryEngine {

    /*
     * Processes a query specification generating a document
     * result set
     * @param spec the query specification
     * @return the document result set
     */
    public ResultSet process(QuerySpecification spec) {
        String terms = spec.getTerms();
        
        if (terms == null || "".equals(terms)) {
            return Search.EMPTY_RESULT_SET;
        }
        
        List filters = new ArrayList();
        addFilters(filters, spec.getFilters());
        
        LuceneSearch search = null;
        if (filters.size() == 0) {
            search = new LuceneSearch(terms);
        } else {
            search = new LuceneSearch(
                terms, 
                new IntersectionFilter((Filter[])filters
                                      .toArray(new Filter[filters.size()])));
        }
        
        return new LuceneResultSet(search);
    }
    
    protected void addFilters(List list,
                              FilterSpecification[] filters) {
        for (int i = 0 ; i < filters.length ; i++) {
            addFilter(list, filters[i]);
        }
    }
    
    protected void addFilter(List list,
                             FilterSpecification filter) {
        FilterType type = filter.getType();
        
        if (ObjectTypeFilterType.KEY.equals(type.getKey())) {
            addObjectTypeFilter(list, (ObjectTypeFilterSpecification)filter);
        }
    }
    
    protected void addObjectTypeFilter(List list,
                                       ObjectTypeFilterSpecification filter) {
        List l = new ArrayList();
        ObjectType[] types = filter.getTypes();
        if (types == null || types.length == 0) {
            return;
        }

        Filter[] filters = new Filter[types.length];
        for (int i = 0 ; i < types.length ; i++) {                        
            filters[i] = new ObjectTypeFilter(types[i].getQualifiedName());
        }
        if (filter.isExclusion()) {
            list.add(new NegationFilter(new UnionFilter(filters)));
        } else {
            list.add(new UnionFilter(filters));
        }
    }    
}
