/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.mimetypes.converters;

import com.arsdigita.db.ConnectionManager;
import com.arsdigita.db.Sequences;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;

import java.math.BigDecimal;
import java.sql.CallableStatement;
import java.sql.SQLException;
import java.sql.Types;

import org.apache.log4j.Logger;

/**
 * A utility class that provides a method for converting document formats.
 *
 * @author Jeff Teeters
 *
 * @version $Revision: #11 $ $DateTime: 2004/04/07 16:07:11 $
 */
public class ConvertFormat {

    private static final Logger s_log = Logger.getLogger(ConvertFormat.class);

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.cms.ConvertFormat";

    private ConvertFormat() {}

    /**
     * Converts a document to html format.  Uses INSO filtering that comes with
     * interMedia.  This allows converting Word, RTF and many other documents to
     * HTML.
     *
     * @param doc_in The document to be converted to html.
     *
     * @return The converted document or <code>null</code> if the conversion
     * could not be done.
     */
    public static String toHTML(byte [] doc_in) {
        // Save document in pre_convert_html table
        java.sql.Connection con = null;
        String returnValue = null;

        BigDecimal id = null;
        // Remove any previously stored entry in pre_convert_html table
        try {
            id = Sequences.getNextValue();
            try {
                new PreConvertHTML(new OID(PreConvertHTML.BASE_DATA_OBJECT_TYPE,
                                           id)).delete();
            } catch (DataObjectNotFoundException e) {
                // good, not found
            }

            // Store document in pre_convert_html and try to convert it.
            PreConvertHTML pre = new PreConvertHTML();
            pre.setId(id);
            pre.setContent(doc_in);
            pre.save();

            // Use interMedia ctx_doc.filter to convert to HTML.
            // See file cms/sql/oracle-se/convert-html.sql
            con = SessionManager.getSession().getConnection();
            CallableStatement funCall =
                con.prepareCall("{ ? = call convert_to_html(" + id + ")}");
            funCall.registerOutParameter(1, Types.VARCHAR);
            funCall.execute();
            returnValue = funCall.getString(1);
            funCall.close();
        } catch (SQLException ex) {
            s_log.error("PreConvertHTML.toHTML failed.", ex);
            return null;
        }

        if (returnValue != null && returnValue.length() > 0) {
            s_log.error("PreConvertHTML.toHTML was unable to convert " +
                        "document with id =" + id + ".  Perhaps its format " +
                        "is not supported.  Error message is: " + returnValue,
                        new Throwable());
        } else {
            try {
                PostConvertHTML pc = new PostConvertHTML
                    (new OID(PostConvertHTML.BASE_DATA_OBJECT_TYPE, id));
                String doc_out = pc.getContent();
                pc.delete();
                s_log.debug ("Read from postConvertHTML = " + doc_out);
                return doc_out;
            } catch (DataObjectNotFoundException ex) {
                s_log.error("PreConvertHTML.toHTML converted doc to html, " +
                            "but unable to retrieve it.  id=" + id, ex);
            }
        }
        return null;
    }
}
