/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.mimetypes;

import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.OID;


/**
 * Used to initialize the table cms_mime_extensions (which contains
 * mime type extensions and the corresponding mime type).
 *
 * @author Jeff Teeters (teeters@arsdigita.com)
 *
 * @version $Revision: #7 $ $DateTime: 2004/04/07 16:07:11 $
 */
public class MimeTypeExtension extends DomainObject {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/mimetypes/MimeTypeExtension.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.cms.MimeTypeExtension";

    public static final String MIME_TYPE = "mimeType";
    public static final String FILE_EXTENSION = "fileExtension";

    public static MimeTypeExtension create(String fileExtension, 
                                           String mimeType) {
        MimeType mimeTypeObject = MimeType.loadMimeType(mimeType);
        if (mimeTypeObject == null) {
            throw new DataObjectNotFoundException
                ("Unable to locate mime type of " + mimeType);
        }
        return create(fileExtension, mimeTypeObject);
    }

    public static MimeTypeExtension create(String fileExtension, 
                                           MimeType mimeType) {
        MimeTypeExtension ext = new MimeTypeExtension();
        ext.setFileExtension(fileExtension);
        ext.setMimeTypeObject(mimeType);
        return ext;
    }

    /**
     *  This retrieves the MimeTypeExtension for the given object
     *  or returns null if none exists
     */
    public static MimeTypeExtension retrieve(String fileExtension) {
        try { 
            return new MimeTypeExtension(new OID(BASE_DATA_OBJECT_TYPE, 
                                                 fileExtension.toLowerCase()));
        } catch (DataObjectNotFoundException e) {
            // there is nothing specified for this mime type
            return null;
        }
    }

    protected MimeTypeExtension() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    public MimeTypeExtension(OID oid) throws DataObjectNotFoundException {
        super(oid);
    }

    public MimeTypeExtension(DataObject obj) {
        super(obj);
    }

    protected MimeTypeExtension(String type) {
        super(type);
    }

    public MimeType getMimeTypeObject() {
        DataObject object = (DataObject)get(MIME_TYPE);
        if (object != null) {
            return new MimeType(object);
        } else {
            return null;
        }
    }

    public void setMimeTypeObject(MimeType object) {
        set(MIME_TYPE, object);
    }

    /**
     *  @deprecated use getMimeTypeObject().getMimeType()
     */
    public String getMimeType() {
        return getMimeTypeObject().getMimeType();
    }

    /**
     *  @deprecated use getMimeTypeObject
     */
    public void setMimeType(String value) {
        MimeType mimeType = MimeType.loadMimeType(value);
        setMimeTypeObject(mimeType);
    }

    public String getFileExtension() {
        return (String) get(FILE_EXTENSION);
    }

    public void setFileExtension(String value) {
        set(FILE_EXTENSION, value.toLowerCase());
    }
}
