/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.logging;

/**
 * <p>
 * Defines APIs for logging in addition to those supplied by log4j.
 * Mainly, this class defines an API for logging securely.
 * </p>
 *
 * @author Yon Feldman 
 * @version $Revision: #7 $ $Date: 2004/04/07 $
 */
public class Log {
    public final static String versionId = "$Id: //core-platform/dev/src/com/arsdigita/logging/Log.java#7 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    // this variable will hold the secure state for this class on a per thread
    // basis.
    private static ThreadLocal s_secureState = new ThreadLocal() {
            public Object initialValue() {
                return Boolean.FALSE;
            }
        };

    /**
     * Checks whether we are currently logging securely.
     *
     * @return <code>true</code> if we are logging securely, <code>false</code> otherwise.
     */
    public static boolean isSecure() {
        return ((Boolean) s_secureState.get()).booleanValue();
    }

    // begin logging securely.
    private static void startSecureLogging() {
        s_secureState.set(Boolean.TRUE);
    }

    // reset the secure logging state to the value passed in.
    private static void resetSecureLogging(Boolean inSecureState) {
        s_secureState.set(inSecureState);
    }

    /**
     * Any log messages run inside this method will be logged securely.
     *
     * @param r the class that implements the <code>Runnable</code> interface
     */
    public static void secure (Runnable r) {
        // store old value so that we can reset to the appropriate value when
        // done.
        boolean inSecureState = isSecure();

        // start logging securely
        startSecureLogging();

        // run code
        r.run();

        // reset logging state
        resetSecureLogging(new Boolean(inSecureState));
    }
}
