/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel.security;

import java.util.Map;
import javax.security.auth.Subject;
import javax.security.auth.callback.CallbackHandler;
import javax.security.auth.login.LoginException;
import org.apache.log4j.Logger;

/**
 * Authenticates a user (loads a user ID) from a credential stored in the
 * current HTTP request.
 *
 * @author Sameer Ajmani
 **/
public abstract class UserLoginModule extends CredentialLoginModule {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/kernel/security/UserLoginModule.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    private static final Logger s_log =
        Logger.getLogger(UserLoginModule.class.getName());

    /**
     * Name of the non-secure user authentication credential.
     **/
    public static final String NORMAL_CREDENTIAL_NAME
        = "ad_user_login";

    /**
     * Name of the secure user authentication credential.
     **/
    public static final String SECURE_CREDENTIAL_NAME
        = "ad_user_login_secure";

    // fields set by initialize()
    private Subject m_subject;
    private CallbackHandler m_handler;
    private Map m_shared;
    private Map m_options;

    public UserLoginModule(CredentialManager manager) {
        super(manager);
    }

    // implements LoginModule
    public void initialize(Subject subject,
                           CallbackHandler handler,
                           Map shared,
                           Map options) {
        super.initialize(subject, handler, shared, options);
        m_subject = subject;
        m_handler = handler;
        m_shared  = shared;
        m_options = options;
    }

    /**
     * Returns the name of the credential.
     *
     * @return <code>SECURE_CREDENTIAL_NAME</code> if the current request is
     * secure, otherwise returns <code>NORMAL_CREDENTIAL_NAME</code>.
     **/
    protected String getCredentialName()
        throws LoginException {
        if (isSecure()) {
            return SECURE_CREDENTIAL_NAME;
        } else {
            return NORMAL_CREDENTIAL_NAME;
        }
    }

    /**
     * Returns the lifetime of the credential in milliseconds.
     *
     * @return <code>FOREVER_SECS</code> in milliseconds if the user
     * requests permanent login, otherwise returns <code>TIMEOUT_SECS</code>
     * in milliseconds.
     **/
    protected long getLifetime()
        throws LoginException {
        return 1000 * (getForever() ? FOREVER_SECS : TIMEOUT_SECS);
    }

    /**
     * Determines whether the user's session should be invalidated.
     *
     * @param value the new value for the credential
     *
     * @return true if the credential is set and has the wrong value.
     **/
    protected boolean shouldInvalidateSession(String value)
        throws LoginException {
        return credentialIsSet() && !credentialHasValue(value);
    }
}
