/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel.permissions;

import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataOperation;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.PersistenceException;
import com.arsdigita.persistence.SessionManager;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;

/**
 * Describes a privilege that can be granted or checked.
 *
 * @author Oumi Mehrotra
 * @version 1.0
 **/
public class PrivilegeDescriptor {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/kernel/permissions/PrivilegeDescriptor.java#16 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static final String ADMIN_NAME = "admin";
    private static final String EDIT_NAME = "edit";
    private static final String READ_NAME = "read";
    private static final String WRITE_NAME = "write";
    private static final String CREATE_NAME = "create";
    private static final String DELETE_NAME = "delete";

    // CMS Privileges...this is part of the hack for the s_implications below
    private final static String CMS_STAFF_ADMIN    = "cms_staff_admin";
    private final static String CMS_CATEGORY_ADMIN = "cms_category_admin";
    private final static String CMS_PUBLISH        = "cms_publish";
    private final static String CMS_NEW_ITEM       = "cms_new_item";
    private final static String CMS_EDIT_ITEM      = "cms_edit_item";
    private final static String CMS_DELETE_ITEM    = "cms_delete_item";
    private final static String CMS_READ_ITEM      = "cms_read_item";
    private final static String CMS_PREVIEW_ITEM   = "cms_preview_item";
    private final static String CMS_CATEGORIZE_ITEMS = "cms_categorize_items";

    /**
     * The PrivilegeDescriptor corresponding to the primitive admin privilege
     */
    public static final PrivilegeDescriptor ADMIN  = new PrivilegeDescriptor(ADMIN_NAME);

    /**
     * The PrivilegeDescriptor corresponding to the read and write privilege
     **/
    public static final PrivilegeDescriptor EDIT =
        new PrivilegeDescriptor(EDIT_NAME);

    /**
     * The PrivilegeDescriptor corresponding to the primitive read privilege
     */
    public static final PrivilegeDescriptor READ =
        new PrivilegeDescriptor(READ_NAME);

    /**
     * The PrivilegeDescriptor corresponding to the primitive write privilege
     */
    public static final PrivilegeDescriptor WRITE =
        new PrivilegeDescriptor(WRITE_NAME);

    /**
     * The PrivilegeDescriptor corresponding to the primitive create privilege
     */
    public static final PrivilegeDescriptor CREATE =
        new PrivilegeDescriptor(CREATE_NAME);

    /**
     * The PrivilegeDescriptor corresponding to the primitive delete privilege
     */
    public static final PrivilegeDescriptor DELETE =
        new PrivilegeDescriptor(DELETE_NAME);

    private String m_displayName;

    private static Map s_privs = Collections.synchronizedMap(new HashMap());

    private String m_name;
    
    private String m_columnName;


    /**
     *
     * Adds a privilege to the system.
     *
     * @exception PersistenceException when there is a persistence
     * error in saving the new privilege.
     *
     * @see #get(String)
     */
    public static PrivilegeDescriptor createPrivilege(String name)
        throws PersistenceException
    {
        if (get(name)!=null) {
            throw new
                IllegalArgumentException("Privilege " + name + " already exists");
        }
        DataObject priv = SessionManager.getSession()
            .create("com.arsdigita.kernel.permissions.Privilege");
        priv.set("privilege", name);
        priv.save();
        addChildPrivilege(ADMIN_NAME, name);
        PrivilegeDescriptor desc = new PrivilegeDescriptor(name);
        put(desc);
        return desc;
    }

    public static void addChildPrivilege(
      String privilegeName,
      String childPrivilegeName) {
      DataOperation addOp =
        SessionManager.getSession().retrieveDataOperation(
          "com.arsdigita.kernel.permissions.addChildPrivilege");
      addOp.setParameter("privilege", privilegeName);
      addOp.setParameter("childPrivilege", childPrivilegeName);
      addOp.execute();
      synchronized (s_privs) {
        s_privs.remove(privilegeName);
        PrivilegeDescriptor priv = new PrivilegeDescriptor(privilegeName);
        s_privs.put(privilegeName, priv);

      }
    }
    
    /**
     *
     * Given a privilege name, returns a privilege descriptor or null
     * if the privilege does not exist on the system.
     *
     * @param privilegeName the name of the privilege to return
     * @return a privilege descriptor, or null if the privilege does not exist.
     *
     * @see #createPrivilege(String)
     */
    public static PrivilegeDescriptor get(String privilegeName) {
        return (PrivilegeDescriptor) s_privs.get(privilegeName);
    }

    /**
     *
     * Returns a collection of privilege descriptors for every privilege in
     * the system.
     * @return a collection of privilege descriptors.
     *
     * @see #get(String)
     */
    public static Collection getAll() {
        return s_privs.values();
    }

    /**
     *
     * Deletes the privilege described by this from the system.
     *
     * @exception PersistenceException when there is a persistence
     * error in saving the new privilege.
     *
     * @see #get(String)
     */
    public void deletePrivilege()
        throws PersistenceException
    {
        OID oid = new OID("com.arsdigita.kernel.permissions.Privilege",
                          m_name);

        DataObject priv = SessionManager.getSession()
            .retrieve(oid);
        priv.delete();
        s_privs.remove(m_name);
    }

    /**
     * Returns the privilege name.
     *
     * @return the privilege name.
     **/
    public final String getName() {
        return m_name;
    }

    /**
     * <br>
     *
     * Returns the display name for the privilege, or just the
     * privilege name if no display name is defined.
     *
     * @return the display name
     */

    public String getDisplayName() {
        if (m_displayName != null) {
            return m_displayName;
        } else {
            return m_name;
        }
    }

    /**
     * Equivalent to getName().
     * @return the privilege name
     */
    public String toString() {
        return getName();
    }

    /**
     * Get column name from privilege - pd_priv_xxx column name mapping.
     * @return the column name from dnm_privilege_col_map table
     */
    public String getColumnName() {
        return m_columnName;
    }
    
    
    /**
     * Determines whether this PrivilegeDescriptor is equal to some other
     * PrivilegeDescriptor. Equality is based on privilege name.
     * @return <code>true</code> if the privilege descriptors are equal;
     * <code>false</code> otherwise.
     */
    public boolean equals(Object o) {
        if (! (o instanceof PrivilegeDescriptor))  {
            return false;
        }
        if (m_name == null) {
            return false;
        }
        PrivilegeDescriptor p = (PrivilegeDescriptor) o;

        return m_name.equals(p.getName());
    }

    public int hashCode() {
        return m_name.hashCode();
    }

    /**
     * Create a new privilege descriptor for use with PermissionDescriptor
     * and PermissionService.
     *
     * @deprecated see #get
     * @param name The name of the privilege.
     **/
    public PrivilegeDescriptor(String name) {
        m_name = name;
        
        DataQuery query = SessionManager.getSession() 
            .retrieveQuery("com.arsdigita.kernel.permissions.PrivilegeColumnNameMap");
        query.setParameter("privilege", name);
        if (!query.next()) {
           query.close();
           throw new 
               RuntimeException("Couldn't find column name for privilege - " + name );
        } else {
            m_columnName = (String)query.get("columnName");
        }
        query.close();
      
    }

    /**
     *
     * Puts a privilege descriptor into the internal cache that is
     * used by the get method.  The put method supports extendibility by allowing
     * subclasses to be returned by the get method.
     */
    protected static void put(PrivilegeDescriptor privDesc) {
        s_privs.put(privDesc.getName(), privDesc);
    }

    /**
     *
     * Returns the list of privilege names that imply this privilege.
     * @return a collection of the privilege names that imply this privilege.
     */
    public Collection getImplyingPrivilegeNames() {
        Collection impliedPrivileges = new HashSet();
        DataQuery query = SessionManager.getSession()
            .retrieveQuery("com.arsdigita.kernel.permissions.ImpliedPrivilege");
        query.setParameter("childPrivilege", m_name);
        while (query.next()) {
        	impliedPrivileges.add((String)query.get("privilege"));
        }
        return impliedPrivileges;
    }

    /**
     *
     * Determine whether or not this privilege is implied by the
     * specified privilege.
     * @return true if the privilege on which this method is invoked
     * is implied by the specified privilege, false otherwise
     */
    public boolean isImpliedBy(PrivilegeDescriptor privilege) {
        return getImplyingPrivilegeNames().contains(privilege.getName());
    }

    /* this is only necessary ...
     * 1. to support the deprectated constructor.  Someone could construct
     *    a privilege named X that has different state than
     *    PrivilegeDescriptor.get(X).  So we make the methods in this class
     *    rely on getCanonical rather than this.
     * 2. The static constants ADMIN, READ, etc. have to be created before
     *    s_privs is fully initialized (for example, s_privs may be initialized
     *    in client code that extends PrivilegeDescriptor).  Therefore,
     *    ADMIN, READ, etc. are special cases of #1 above.
     */
    private PrivilegeDescriptor getCanonical() {
        return get(this.getName());
    }

    /**
     * Initializes the PrivilegeDescriptor's internal cache of privileges.
     * Called from the kernel initializer.
     */
    public static void initialize() {
        DataCollection privs = SessionManager.getSession()
            .retrieve("com.arsdigita.kernel.permissions.Privilege");
            while (privs.next()) {
                String name = (String) privs.get("privilege");
                final PrivilegeDescriptor desc = new PrivilegeDescriptor(name);
                put(desc);
            }
        privs.close();

    }

}
