/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import com.arsdigita.domain.DomainObjectInstantiator;
import com.arsdigita.domain.DomainObjectFactory;
import java.util.List;
import java.util.Iterator;
import java.util.ArrayList;
import org.apache.log4j.Logger;

/**
 * <p>Automates the creation and setup of ResourceTypes.</p>
 *
 * @author Jim Parsons &lt;<a href="mailto:jparsons@redhat.com">jparsons@redhat.com</a>&gt;
 */
public class ResourceSetup {

    protected Logger m_logger;
    protected String m_title = null;
    protected String m_description = null;
    protected String m_typeName = null;
    protected DomainObjectInstantiator m_instantiator = null;

    public ResourceSetup(Logger logger) {
        m_logger = logger;
    }



    public void setTitle(String title) {
        m_title = title;
    }

    public void setDescription(String description) {
        m_description = description;
    }

    public void setResourceObjectType(String typeName) {
        m_typeName = typeName;
    }

    public void setInstantiator(DomainObjectInstantiator instantiator) {
        m_instantiator = instantiator;
    }

    protected void notice(String message) {
        m_logger.info("ResourceType '" + m_title + "' - " + message);
    }

    public ResourceType run() {
        notice("Validating setup...");

        List messages = validate();

        if (messages.size() > 0) {
            Iterator iter = messages.iterator();

            while (iter.hasNext()) {
                m_logger.error((String)iter.next());
            }

            return null;
        }

        notice("Done validating.");

        ResourceType resourceType = process();

        if (resourceType != null) {
            resourceType.save();
        }

        return resourceType;
    }

    protected List validate() {
        ArrayList messages = new ArrayList();

        if (m_title == null)
            messages.add("Title is not set.");
        if (m_typeName == null)
            messages.add("ResourceObjectType is not set.");
        if (m_instantiator == null)
            messages.add("Instantiator is not set.");

        return messages;
    }

    protected ResourceType process() {
        notice("Starting Resource setup...");

        ResourceType resourceType = null;

        if (!ResourceType.isInstalled(m_typeName)) {

            notice("Not installed.  Installing now...");

            notice("Using the following properties to perform install.");
            notice("  ResourceObjectType: " + m_typeName);
            notice("  Title: " + m_title);
            notice("  Description: " + m_description);
            notice("  Instantiator: " + m_instantiator);

            resourceType = ResourceType.createResourceType(m_title,m_typeName);

            resourceType.setDescription(m_description);

            notice("Done installing.");
        }

        else {

        resourceType = ResourceType.retrieveResourceTypeForResource(m_typeName);

        }

        DomainObjectFactory.registerInstantiator(m_typeName, m_instantiator);

        notice("Done setting up Resource.");

        return resourceType;


    }

}
