/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.kernel;

import org.apache.log4j.Logger;
import java.util.Locale;

/**
 *
 * A kernel excursion is a way of making your code execute under an
 * alternative environment (context). Override the excurse method to
 * create a KernelExcursion. For example:
 *
 * <code>
 *      KernelExcursion rootExcursion = new KernelExcursion() {
 *              public void excurse() {
 *                  // Set up specific context variables.
 *                  setEffectiveParty(Kernel.getSystemParty());
 *
 *                  // Execute code in new context.
 *
 *              }};
 *
 *      rootExcursion.run();
 * </code>
 */
public abstract class KernelExcursion implements Runnable {
    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/kernel/KernelExcursion.java#18 $" +
        "$Author: dennis $" +
        "$DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log = Logger.getLogger
        (KernelExcursion.class);

    public final void run() {
        s_log.debug("Running excursion");

        KernelContext context = Kernel.getContext();

        Kernel.setContext(context.copy());

        try {
            excurse();
        } finally {
            Kernel.setContext(context);
        }
    }

    /**
     * <p>When this method is called the current KernelContext is
     * copied and the code inside the excurse() method is given the
     * opportunity to modify the new context. Any code then called
     * from within the excurse() method is executed in this new
     * context. After the excurse() method completes the old context
     * is restored exactly as it was before. This makes it unnecessary
     * to write error prone code like this:</p>
     *
     * <blockquote><pre>
     *   Party oldParty = context.getParty();
     *   context.setParty(newParty);
     *   ...
     *   // do something
     *   ...
     *   context.setParty(oldParty); // If this is forgotten, bad
     *                               // things can happen.
     * </pre></blockquote>
     */
    protected abstract void excurse();

    protected final void setEffectiveParty(Party party) {
        Kernel.getContext().setEffectiveParty(party);
    }

    protected final void setParty(Party party) {
        Kernel.getContext().setParty(party);
    }

    protected final void setResource(Resource app) {
        Kernel.getContext().setResource(app);
    }

    protected final void setLocale(Locale locale) {
        Kernel.getContext().setLocale(locale);
    }

    protected final void setSessionID(String sessionID) {
        Kernel.getContext().setSessionID(sessionID);
    }

    protected final void setTransaction(DatabaseTransaction transaction) {
        Kernel.getContext().setTransaction(transaction);
    }
}
