/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.formbuilder;


import com.arsdigita.formbuilder.PersistentLabel;
import com.arsdigita.formbuilder.PersistentWidget;

import com.arsdigita.domain.DataObjectNotFoundException;

import com.arsdigita.formbuilder.util.FormBuilderUtil;

import com.arsdigita.kernel.ACSObject;

import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.DataCollection;
import com.arsdigita.persistence.DataObject;

import com.arsdigita.persistence.OID;

import java.math.BigDecimal;

/**
 * This class is used to associate {@link com.arsdigita.formbuilder.PersistentLabel}s
 * with {@link com.arsdigita.formbuilder.PersistentWidget}s.
 *
 * This association facilitates the UI processes for
 * adding, deleting and moving widgets on forms.
 */
public class WidgetLabel extends PersistentLabel {
    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.formbuilder.WidgetLabel";

    public static String WIDGET_ID = "widgetId";

    /**
     * Constructor. Creates a new widget label.
     */
    public WidgetLabel() {
        this(BASE_DATA_OBJECT_TYPE);
    }

    /**
     * Constructor. Used by subclasses to specify
     * a different base data object type
     *
     * @param typeName the base data object type
     */
    public WidgetLabel(String typeName) {
        super(typeName);
    }

    /**
     * Constructor. Used by subclasses to specify
     * a different base data object type
     *
     * @param type the object type
     */
    public WidgetLabel(com.arsdigita.persistence.metadata.ObjectType type) {
        super(type);
    }

    /**
     * Constructor. Instantiates a widget label
     * from a previously retrieved data object.
     *
     * @param obj the data object
     */
    public WidgetLabel(DataObject obj) {
        super(obj);
    }

    /**
     * Constructor. Instantiates a widget label
     * retrieving the the data object with the
     * specified id
     *
     * @param id the id of the widget label to retrieve
     */
    public WidgetLabel(BigDecimal id)
        throws DataObjectNotFoundException {

        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    /**
     * Constructor. Used by subclasses to retrieve
     * an existing widget label matching the
     * supplied oid.
     *
     * @param oid the oid of the widget label to retrieve
     */
    public WidgetLabel(OID oid)
        throws DataObjectNotFoundException {

        super(oid);
    }

    /**
     * This method creates a new widget and initialises
     * all the required attributes
     *
     * @param widget the persistent widget associated with the label
     * @param label the text for the label
     */
    public static WidgetLabel create(PersistentWidget widget,
                                     String label) {
        WidgetLabel l = new WidgetLabel();

        l.setLabel(label);
        l.setWidget(widget);

        return l;
    }

    /**
     * Retrieves the widget label class associated with
     * the specified persistent widget.
     *
     * @param widget the persistent widget whose label to find
     * @throws com.arsdigita.domain.DataObjectNotFoundException if the
     * there is no label associated with the widget
     */
    public static WidgetLabel findByWidget(PersistentWidget widget)
        throws DataObjectNotFoundException {

        Session ssn = SessionManager.getSession();
        DataCollection labels = ssn.retrieve(BASE_DATA_OBJECT_TYPE);
        labels.addEqualsFilter(WIDGET_ID, widget.getID());

        if (labels.next()) {
            DataObject obj = labels.getDataObject();
            WidgetLabel l = new WidgetLabel(obj);
            labels.close();
            return l;
        } else {
            throw new DataObjectNotFoundException("cannot find widget label");
        }
    }

    /**
     * Sets the persistent widget associated with this label
     */
    public void setWidget(PersistentWidget widget) {
        set(WIDGET_ID, widget.getID());
    }

    /**
     * Sets the id of the persistent widget associated with this label
     */
    public void setWidgetID(BigDecimal id) {
        set(WIDGET_ID, id);
    }

    /**
     * Retrieves the persistent widget object associated
     * with this label. Using this method is not very
     * desirable since it has to do one query to find out
     * the default domain class, and another to actually
     * retrieve the object.
     */
    public PersistentWidget getWidget()
        throws DataObjectNotFoundException {
        // XXX how the fsck do I retireve an object without knowing is data type ?

        BigDecimal id = (BigDecimal)get(WIDGET_ID);

        OID oid = new OID(ACSObject.BASE_DATA_OBJECT_TYPE, id);
        Session s = SessionManager.getSession();
        DataObject o = s.retrieve(oid);

        return (PersistentWidget)FormBuilderUtil.instantiateObjectOneArg((String)o.get("defaultDomainClass"), id);
    }

    /**
     * Retrieves the id of the persistent widget associated
     * with this label
     */
    public BigDecimal getWidgetID() {
        return (BigDecimal)get(WIDGET_ID);
    }
}
