/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.formbuilder;


// All ACSObjects have a unique id
import java.math.BigDecimal;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.metadata.ObjectType;

// This domain object can create a Bebop form
import com.arsdigita.bebop.Form;


// Thrown if the underlying DataObject with given id cannot be found
import com.arsdigita.domain.DataObjectNotFoundException;

// All PersistentComponentFactories can create a Bebop Component
import com.arsdigita.bebop.Component;

import com.arsdigita.formbuilder.util.FormBuilderUtil;

// ACS 5 uses Log4J for logging
import org.apache.log4j.Logger;


/**
 * This domain object manages persistence of a Bebop Form. Properties
 * and components of a Bebop Form can be set (typically with an admin interface),
 * the form should then be saved so that it can be resurrected later
 * on in the application. Of particular importance to application
 * developers are the methods in the PersistentComponentFactory and
 * PersistentContainerFactory interfaces.
 *
 * @author Peter Marklund
 * @version $Id: //core-platform/dev/src/com/arsdigita/formbuilder/PersistentForm.java#11 $
 *
 */
public class PersistentForm extends PersistentFormSection {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/formbuilder/PersistentForm.java#11 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private static final Logger s_log =
        Logger.getLogger(PersistentForm.class.getName());

    /**
     * BASE_DATA_OBJECT_TYPE represents the full name of the
     * underlying DataObject of this class.
     */
    public static final String BASE_DATA_OBJECT_TYPE =
        "com.arsdigita.formbuilder.FormSection";

    // *** Constructors -------------

    /**
     * Default Constructor. Should normally not be used be application
     * developers. Is only here to enable instantiation by FormSectionGenerator
     */
    public PersistentForm() {
        super(BASE_DATA_OBJECT_TYPE);
    }

    /**
     * Constructor that creates a new form domain object that
     * can be saved to the database later on.
     */
    public PersistentForm(String typeName) {
        super(typeName);
    }

    public PersistentForm(ObjectType type) {
        super(type);
    }

    public PersistentForm(DataObject obj) {
        super(obj);
    }

    /**
     * Constructor that retrieves an existing form domain object
     * from the database.
     *
     * @param id The object id of the form domain object to retrieve
     */
    public PersistentForm(BigDecimal id)
        throws DataObjectNotFoundException {

        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    public PersistentForm(OID oid)
        throws DataObjectNotFoundException {

        super(oid);
    }

    public static PersistentForm create(String htmlName) {
        PersistentForm f = new PersistentForm();
        f.setup(htmlName);
        return f;
    }

    protected void setup(String htmlName) {
        setHTMLName(htmlName);
    }

    /**
     * Create a Bebop form using the persistent information in this form domain object.
     *
     */
    public Component createComponent() {

        // Create the form with the name attribute
        Form form;
        if (getFormContainer() != null) {
            form = new Form(getHTMLName(), getFormContainer());
        } else {
            form = new Form(getHTMLName());
        }

        // Set the action attribute
        form.setAction(getAction());

        // Add the process listener
        addProcessListeners(form);

        // Add the components
        addComponents(form);

        return form;
    }

    // *** Attribute Methods

    public void setHTMLName(String name) {
        setComponentAttribute("name", name);
    }

    public String getHTMLName() {
        return getComponentAttribute("name");
    }

    public void setAction(String action) {
        set("action", action);
    }

    public String getAction() {
        return (String)get("action");
    }

}
