/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.domain;

import com.arsdigita.persistence.DataObject;

import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 *
 * This instantiator is primarily intended for other instantiators
 * (such as ACSObjectInstantiator) to provide the default behavior
 * of instiating a domain class using reflection.
 **/
public class ReflectionInstantiator extends DomainObjectInstantiator {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/domain/ReflectionInstantiator.java#10 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    private static Logger s_log = Logger.getLogger(ReflectionInstantiator.class);

    private Constructor m_constructor;

    /**
     * keyed off of class name.
     **/
    private static Map s_instantiators = new HashMap();

    private static Class[] s_dataArgs = new Class[]{DataObject.class};

    public static ReflectionInstantiator
        getInstantiator(String domainClassName)
    {
        ReflectionInstantiator instantiator = null;
        synchronized (ReflectionInstantiator.class) {
            if (s_instantiators.containsKey(domainClassName)) {
                instantiator =
                    (ReflectionInstantiator) s_instantiators.get(domainClassName);
            } else {
                try {
                    instantiator = new ReflectionInstantiator(domainClassName);
                } catch (Exception e) {
                    s_log.warn("cannot create reflection instantiator for " + domainClassName, e);
                    // no class with specified name and public constructor
                    // of form Constructor(DataObject).  Just return null.
                }
                s_instantiators.put(domainClassName, instantiator);
            }
        }
        return instantiator;
    }

    private ReflectionInstantiator(String domainClassName)
        throws LinkageError,
               ExceptionInInitializerError,
               ClassNotFoundException,
               NoSuchMethodException,
               SecurityException
    {
        Class javaClass = Class.forName(domainClassName);
        m_constructor = javaClass.getConstructor(s_dataArgs);
        // We should add a check that the domain class is indeed
        // a subclass of DomainObject
    }

    /**
     * Construct a DomainObject given a data object.  Called from
     * DomainObjectFactory.newInstance() as the last step of
     * instantiation.
     *
     * @param dataObject The data object from which to construct a domain
     * object.
     *
     * @return A domain object for this data object, or null if unable to create.
     */
    protected DomainObject doNewInstance(DataObject dataObject) {
        try {
            return (DomainObject)
                m_constructor.newInstance(new DataObject[]{dataObject});
        } catch (InstantiationException ex) {
            return reportError(dataObject, ex);
        } catch (IllegalAccessException ex) {
            return reportError(dataObject, ex);
        } catch (InvocationTargetException ex) {
            return reportError(dataObject, ex);
        }
    }

    private DomainObject reportError(DataObject dobj, Throwable ex) {
        s_log.error("Construtor " + m_constructor +
                    " could not create a new DomainObject " +
                    "for DataObject: " + dobj, ex);
        return null;
    }
}
