/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.dispatcher;

import java.util.Locale;
import java.util.ResourceBundle;
import javax.servlet.ServletContext;

/**
 * Interface used when dispatchers are
 * chained or piped together. Part of the requested URL will be used
 * at each stage of the dispatch, and this interface is used by the
 * dispatcher to tell what part of the URL has already been used to
 * dispatch the request so far. The remainder is what the current
 * dispatcher must work with.  Because form/URL variables are not
 * order-dependent, We only keep track of the path portion
 * of the URL.
 *
 * @author Bill Schneider 
 * @version $Revision: #9 $ $Date: 2004/04/07 $
 * @since 4.5 */

public interface RequestContext {

    public static final String versionId =
        "$Id: //core-platform/dev/src/com/arsdigita/dispatcher/RequestContext.java#9 $";

    /**
     * Gets the portion of the URL that has not been used by
     * previous dispatchers in the chain.
     * @return the portion of the URL that must be used by
     * the current dispatcher.
     */
    public String getRemainingURLPart();

    /**
     * Gets the portion of the URL that has already been used by
     * previous dispatchers in the chain.
     * @return the portion of the URL that has already been used.
     */
    public String getProcessedURLPart();

    /**
     * Gets the original URL requested by the end user's browser.
     * This URL does <em>not</em> change when a request is forwarded
     * by the application; "/foo/bar" is still the original request
     * URI in the browser even if we've dispatched the request to
     * "/packages/foo/www/bar.jsp".
     *
     * @return the original URL requested by the end user's browser.
     * All generated HREF, IMG SRC, and FORM ACTION attributes, and
     * any redirects, will be relative to this URL.
     */
    public String getOriginalURL();

    /**
     * Gets the current servlet context.
     * @return the current servlet context, which must be set by implementation.
     */
    public ServletContext getServletContext();

    /**
     * more methods will be implemented as needed, for locale,
     * form variables, etc.
     */

    /**
     * Gets the locale for the current request context.
     * @return the locale for the current request context.
     */
    public Locale getLocale();

    /**
     * Returns a <code>java.util.ResourceBundle</code> for the
     * current request, based on the requested application and the
     * user's locale preference.
     *
     * @return the current <code>java.util.ResourceBundle</code> to use
     * in this request.
     */
    public ResourceBundle getResourceBundle();

    /**
     * Gets the requested output type.
     * @return the requested output type (normally "text/html" by default
     * for a web browser request).
     */
    public String getOutputType();

    /**
     * Gets the debugging flag.
     * @return the debugging flag.
     * Currently, debugging applies to XSL transformation.
     */
    public boolean getDebugging();

    /**
     * Gets the show-XML-only flag.
     * @return if true, indicates that the active
     * <code>PresentationManager</code> should output raw, untransformed
     * XML instead of processing it with XSLT.
     */
    public boolean getDebuggingXML();

    /**
     * Gets the show-XSL-only flag.
     * @return if true, indicates that the active
     * <code>PresentationManager</code> should output the XSLT stylesheet
     * in effect for this request.
     */
    public boolean getDebuggingXSL();


    /**
     * Gets the base path, relative to the webapp root, where JSP-based
     * resources (and static pages) will be found.
     * @return the base path, relative to the webapp root, where
     * JSP-based resources will be found.
     * Returns with a trailing slash (for example,
     * /packages/package-key/www/).
     */
    public String getPageBase();
}
