/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.util;

import java.util.Iterator;
import java.util.Set;
import java.util.HashSet;
import com.arsdigita.bebop.Component;
import org.apache.log4j.Logger;

/**
 * <p></p>
 *
 * <p>A utility class for walking down a tree of Bebop components and
 * performing some work on each one.</p>
 *
 * <p>Uses a filter to perform the action only on certain components.
 * This filter may be used to skip only individual components or entire
 * subtrees.  The default filter matches all components.</p>
 *
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/util/Traversal.java#8 $
 */
public abstract class Traversal {
    private static final Logger s_log = Logger.getLogger(Traversal.class);

    /**
     * If <code>test</code> returns <code>PERFORM_ACTION</code>,
     * then the action is performed on the component and its children.
     * (This is the default.)
     */
    public final static int PERFORM_ACTION = 0;

    /**
     * If <code>test</code> returns <code>SKIP_COMPONENT</code>,
     * then the current component is skipped but its descendants are  still
     * traversed.
     */
    public final static int SKIP_COMPONENT = 1;

    /**
     * If <code>test</code> returns <code>SKIP_SUBTREE</code>,
     * then the current component and all of its descendants are skipped.
     */
    public final static int SKIP_SUBTREE = 2;

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/util/Traversal.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private Set m_visiting = null;

    {
        if (s_log.isDebugEnabled()) {
            m_visiting = new HashSet();
        }
    }

    /**
     * Defines the action to be performed on each node.  Users of this
     * class should override this method with behavior for their
     * particular domain.
     *
     * @param c the component on which to perform this action.
     */
    protected abstract void act(Component c);

    /**
     * Invoke {@link #act} on this component, and then do the same for
     * each of its children for which the supplied
     * <code>test</code> condition is true.
     *
     * @param c the component on which to call {@link #act}.  */
    public void preorder(Component c) {
        if (s_log.isDebugEnabled() && m_visiting.contains(c)) {
            s_log.debug("Cycle detected at component " + c +
                        "; visiting nodes: " + m_visiting);
            throw new IllegalStateException
                ("Component " + c + " is part of a cycle");
        }

        int flag = test(c);

        if (flag == PERFORM_ACTION) {
            act(c);
        }

        if (flag != SKIP_SUBTREE) {
            if (s_log.isDebugEnabled()) {
                m_visiting.add(c);
            }

            for (Iterator i = c.children(); i.hasNext(); ) {
                preorder ((Component) i.next());
            }
        }

        if (s_log.isDebugEnabled()) {
            m_visiting.remove(c);
        }
    }

    /**
     * The default component test returns <code>PERFORM_ACTION</code>
     * to act on all components in the tree.  Override this method
     * to supply your own component test.
     * @param c the component to test
     * @return by default returns <code>PERFORM_ACTION</code> on all
     * components.
     */
    protected int test(Component c) {
        return PERFORM_ACTION;
    }
}
