/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.parameters;

import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.parameters.ParameterData;
import com.arsdigita.bebop.event.ParameterListener;
import com.arsdigita.bebop.event.ParameterEvent;

/**
 *     Verifies that the
 *    parameter's value is within a specified range.
 *
 *    @author Karl Goldstein 
 *    @author Uday Mathur 
 *    @author Stas Freidin 
 *    @author Rory Solomon  */
public class NumberInRangeValidationListener implements ParameterListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/parameters/NumberInRangeValidationListener.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private final double m_lowerBound;
    private final double m_upperBound;
    private final String m_baseErrorMsg;

    public NumberInRangeValidationListener(Number a, Number b) {
        this(a.doubleValue(),b.doubleValue());
    }

    public NumberInRangeValidationListener(long lower, long upper) {
        this( (double)lower, (double)upper );
    }

    public NumberInRangeValidationListener(double lower, double upper) {
        if ( upper < lower ) {
            throw new IllegalArgumentException
                ("Lower bound must be less than or equal to upper bound.");
        }

        m_lowerBound = lower;
        m_upperBound = upper;

        StringBuffer msg = new StringBuffer(128);
        msg.append("The following values are out of the specified range of (")
            .append(m_lowerBound)
            .append(",")
            .append(m_upperBound)
            .append(") :");

        m_baseErrorMsg = msg.toString();
    }

    public void validate (ParameterEvent e) throws FormProcessException {
        // note: The abstract class Number is the superclass of classes
        // Byte, Double, Float, Integer, Long, and Short.

        ParameterData data = e.getParameterData();
        Object obj = data.getValue();
        boolean isValid = true;

        // Another listener will validate that these values are present if
        // required, but we don't want any null pointer exceptions.
        if ( obj == null ) return;


        StringBuffer msg = null;
        if ( obj instanceof Number[] ) {
                Number[] values = (Number[]) obj;
                for (int i = 0; i < values.length; i += 1) {
                    double val = values[i].doubleValue();
                    if ( m_lowerBound > val || val > m_upperBound ) {
                        if (msg == null) {
                            msg = new StringBuffer(m_baseErrorMsg);
                        }
                        msg.append(val);
                        msg.append(" ");
                        isValid = false;
                    }
                }
        } else if (obj instanceof Number) {

                Number value = (Number) obj;
                double val = value.doubleValue();
                if ( m_lowerBound > val || val > m_upperBound ) {
                    msg = new StringBuffer(m_baseErrorMsg);
                    msg.append(val);
                    isValid = false;
                }
        } else {
            throw new FormProcessException("Unexpected value type: " + obj.getClass());
        }

        if (!isValid) {
            data.addError(msg.toString());
        }
    }
}
