/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.parameters;

import com.arsdigita.bebop.parameters.ParameterData;
import com.arsdigita.bebop.event.ParameterListener;
import com.arsdigita.bebop.event.ParameterEvent;

/**
 *     Verifies that the
 *    parameter's value is a member of a list of Strings
 *
 *    @author Karl Goldstein 
 *    @author Uday Mathur 
 *    @author Stas Freidin 
 *    @author Rory Solomon  */

public class EnumerationValidationListener implements ParameterListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/parameters/EnumerationValidationListener.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private final String[] m_allowedValues;
    private final boolean m_caseSensitive;
    private final String m_baseErrorMsg;

    public EnumerationValidationListener(String[] allowedValues) {
        this(allowedValues, false);
    }

    public EnumerationValidationListener(String[] allowedValues,
                                         boolean caseSensitive) {
        m_allowedValues = allowedValues;
        m_caseSensitive = caseSensitive;
        StringBuffer msg = new StringBuffer(128);

        msg.append(" must be one of:");
        for (int j = 0; j < m_allowedValues.length;j++) {
            msg.append(m_allowedValues[j]).append(" ");
        }

        m_baseErrorMsg = msg.toString();
    }


    // there are many cool optimizations we plan to put here
    // 1)changing the order of the loops
    // 2)different data structs based on size
    // Another listener will validate that these values are present if
    // required, but we don't want any null pointer exceptions.

    public void validate (ParameterEvent e) {
        /* this loop is theta(n^2), but the constant factor is small.
           using Collection.contains() would also be n^2 with more memory overhead
           Hashmaps are disasters for memory, and good only at large sizes.
        */
        ParameterData data = e.getParameterData();


        if ( m_allowedValues == null) { return; }

        if (data.getValue() instanceof Object[]) {
            String[] names =(String[]) data.getValue();
            for (int i = 0; i < names.length; i += 1) {
                String name = names[i];
                if (!validateOneString(name)) {
                    data.addError(name + m_baseErrorMsg);
                }
            }
        } else {
            String name = (String) data.getValue();
            if (!validateOneString(name)) {
                data.addError(name + m_baseErrorMsg);
            }
        }
    }

    private boolean validateOneString ( String value) {
        if ( value == null ) return true;

        boolean isValid = false;

        for (int j = 0; j < m_allowedValues.length; j++) {
            if (m_caseSensitive) {
                if (value.equals(m_allowedValues[j])) {
                    isValid = true;
                    break;
                }
            } else {
                if (value.equalsIgnoreCase(m_allowedValues[j])) {
                    isValid = true;
                    break;
                }
            }
        }

        return isValid;
    }
}
