/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.parameters;

import java.text.DateFormat;
import java.util.Date;

import com.arsdigita.bebop.parameters.ParameterData;
import com.arsdigita.bebop.event.ParameterListener;
import com.arsdigita.bebop.event.ParameterEvent;

/**
 *     Verifies that the
 *    parameter's value is within a specified date range.
 *
 *    @author Karl Goldstein 
 *    @author Uday Mathur 
 *    @author Stas Freidin 
 *    @author Rory Solomon  */

public class DateInRangeValidationListener implements ParameterListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/parameters/DateInRangeValidationListener.java#9 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
    private final Date m_lowerBound;
    private final Date m_upperBound;
    private final String m_baseErrorMsg;

    public DateInRangeValidationListener(Date lower, Date upper) {
        if ( lower.compareTo(upper) > 0 ) {
            throw new IllegalArgumentException
                ("Lower bound must be earlier than or equal to upper bound.");
        }

        m_lowerBound = lower;
        m_upperBound = upper;

        DateFormat formatter = DateFormat.getDateInstance();
        StringBuffer msg = new StringBuffer(128);
        msg.append("The following dates are out of the specified range of (")
            .append(formatter.format(m_lowerBound))
            .append(",")
            .append(formatter.format(m_upperBound))
            .append(") :");

        m_baseErrorMsg = msg.toString();
    }

    public void validate (ParameterEvent e) {
        // gets the date format for the default style and locale
        DateFormat formatter = DateFormat.getDateInstance();

        ParameterData data = e.getParameterData();
        Object obj = data.getValue();

        // Another listener will validate that these values are present if
        // required, but we don't want any null pointer exceptions.
        if ( obj == null ) return;

        boolean isValid = true;

        StringBuffer msg = null;
        if ( data.getValue() instanceof Object[] ) {
            Date[] values = (Date[]) obj;

            for (int i = 0; i < values.length; i += 1) {
                final Date value = values[i];
                if (isOutOfRange(value)) {
                    if (msg == null) {
                        msg = makeErrorBuffer();
                    }
                    msg.append(" ").append(formatter.format(value));
                    isValid = false;
                }
            }
        } else {
            final Date value = (Date) obj;
            if (isOutOfRange(value)) {
                msg = makeErrorBuffer();
                msg.append(" ").append(formatter.format(value));
                isValid = false;
            }
        }

        if (! isValid) {
            data.addError(msg.toString());
        }
    }

    private boolean isOutOfRange(final Date value) {
        return m_lowerBound.compareTo(value) >= 0
                               || value.compareTo(m_upperBound) >= 0;
    }

    private StringBuffer makeErrorBuffer() {
        return new StringBuffer(m_baseErrorMsg);
    }
}
