/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.jsp;

import com.arsdigita.util.Assert;

/**
 * A wrapper exception that can be used to rethrow another
 * exception.
 *
 * The standard JspException doesn't accept a root cause,
 * hence this subclass.
 *
 * @author David Eison 
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/jsp/JspWrapperException.java#5 $
 */
public class JspWrapperException extends RuntimeException {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/jsp/JspWrapperException.java#5 $";

    Throwable m_rootCause;

    /**
     * Constructor which takes a root cause
     * that this exception will be wrapping.
     */
    public JspWrapperException (Throwable rootCause) {
        this(null, rootCause);
    }

    /**
     * Constructor which takes a message string and a root cause
     * that this exception will be wrapping.  The message string
     * should be something different than rootCause.getMessage()
     * would normally provide.
     */
    public JspWrapperException (String s, Throwable rootCause) {
        super(s);
        Assert.assertNotNull(rootCause);
        this.m_rootCause = rootCause;
    }

    /**
     * Gets the root cause of this exception.
     */
    public Throwable getRootCause() {
        return m_rootCause;
    }

    // All further methods override normal throwable behavior to
    // combine information w/ the root cause.

    /**
     * Get a string representing this exception and the root cause.
     */
    public String toString() {
        return toString(this.getClass());
    }

    /**
     * Get a string representing this exception and the root cause.
     *
     * Functions like normal toString, except that the name of the
     * provided class will be used instead of the name of the
     * unchecked wrapper exception.  Useful when another exception
     * class is using  an unchecked wrapper exception to delegate
     * to.
     */
    public String toString(Class delegatingClass) {
        // default toString calls getMessage, so we don't want to rely on it
        // here.
        StringBuffer b = new StringBuffer(delegatingClass.getName());
        String superMsg = super.getMessage();
        if (superMsg != null) {
            b.append(": ").append(superMsg);
        }
        b.append(" (root cause: ").append(m_rootCause.toString());
        b.append(")");
        return b.toString();
    }

    /**
     * This exception's message and the root cause's.
     */
    public String getMessage() {
        if (m_rootCause != null) {
            return super.getMessage() + " (root cause: " + m_rootCause.getMessage() + ")";
        } else {
            return super.getMessage();
        }
    }

    /**
     * Stack trace for the root cause.
     */
    public void printStackTrace() {
        super.printStackTrace();
        System.err.print("Root cause: ");
        m_rootCause.printStackTrace();
    }

    /**
     * Stack trace for the root cause.
     */
    public void printStackTrace(java.io.PrintStream s) {
        super.printStackTrace(s);
        s.println("Root cause: ");
        m_rootCause.printStackTrace(s);
    }

    /**
     * Stack trace for the root cause.
     */
    public void printStackTrace(java.io.PrintWriter s) {
        super.printStackTrace(s);
        s.println("Root cause: ");
        m_rootCause.printStackTrace(s);
    }
}
