/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop.event;

import java.util.EventListener;

/**
 * Listeners of this class are called just before a {@link com.arsdigita.bebop.Component} is
 * about to be output, either in the form of an XML element, or by printing
 * its HTML representation. The {@link #prepare prepare method} of the
 * listener can make modifications to the {@link PrintEvent#getTarget
 * target} of the event. The target will then be used to produce output
 * instead of the source.
 * <p>
 * {@link PrintEvent PrintEvents} are <i>unicast</i> events, which means
 * that components should only permit the registration of one
 * <code>PrintListener</code>. Since the <code>PrintListener</code> is
 * expected to modify the target, allowing multiple listeners to modify the
 * target of one event would make it impossible to predict the resulting
 * target component, since an individual listener can not know which
 * listeners have run before it and which ones will run after it.
 * <p>
 * As an example consider the following code:
 * <pre>
 *   Label l = new Label("Default text");
 *   l.addPrintListener( new PrintListener {
 *     private static final BigDecimal ONE = new BigDecimal(1);
 *     private BigDecimal count = new BigDecimal(0);
 *     public void prepare(PrintEvent e) {
 *       Label t = e.getTarget();
 *       synchronized (count) {
 *         count.add(ONE);
 *       }
 *       t.setLabel("Call no." + count + " since last server restart");
 *     }
 *   });</pre>
 * Adding the label <code>l</code> to a page will lead to a label that
 * changes in every request and print how many times the containing label
 * has been called.
 *
 * @author Karl Goldstein
 * @author Uday Mathur
 * @author David Lutterkort
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/event/PrintListener.java#8 $
 */

public interface PrintListener extends EventListener {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/event/PrintListener.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    /**
     * Prepare the target component returned by {@link PrintEvent#getTarget
     * e.getTarget()} for output. The target component is an unlocked clone
     * of the source of the event and can be freely modified within this
     * method.
     *
     * @param e Event containing the page state, the source and the target of
     * the event
     *
     * @see PrintEvent
     */

    void prepare(PrintEvent e);

}
