/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop;

import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.form.Hidden;
import com.arsdigita.bebop.form.Widget;
import com.arsdigita.bebop.parameters.BooleanParameter;
import com.arsdigita.bebop.util.Traversal;
import com.arsdigita.xml.Element;

import org.apache.log4j.Logger;

/**
 * The FormStep class modifies the behavior of FormSection with respect to
 * listener firing. Instead of firing init listeners the first time the
 * enclosing form is displayed on the page, the FormStep class fires init
 * listeners the first time the FormStep itself is displayed on the page. The
 * process, validate, and submission listeners are then fired on every
 * submission following the one in which the init listeners were fired. This
 * behavior is useful when used in conjunction with {@link MultiStepForm} or
 * its subclasses to provide initialization in later steps of a multi step
 * form that depends on the values entered in earlier steps.
 *
 * @see Wizard
 * @see MultiStepForm
 *
 * @author <a href="mailto:rhs@mit.edu">rhs@mit.edu</a>
 * @version $Revision: #8 $ $Date: 2004/04/07 $
 **/

public class FormStep extends FormSection {
    private final static Logger s_log = Logger.getLogger(FormStep.class);

    public final static String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/FormStep.java#8 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private Form m_form = null;

    private Hidden m_initialized;

    /**
     * Constructs a new FormStep with the given name. The name must uniquely
     * identify this FormStep within it's enclosing Form.
     *
     * @param name A name that uniquely identifies this FormStep within it's
     *             enclosing Form.
     **/

    public FormStep(String name) {
        addInitialized(name);
    }

    /**
     * Constructs a new FormStep with the given name. The name must uniquely
     * identify this FormStep within it's enclosing Form.
     *
     * @param name A name that uniquely identifies this FormStep within it's
     *             enclosing Form.
     * @param panel The container used to back this FormStep.
     **/

    public FormStep(String name, Container panel) {
        super(panel);
        addInitialized(name);
    }

    protected FormStep(String name, Container panel, FormModel model) {
        super(panel, model);
        addInitialized(name);
    }

    public void register(Page p) {
        super.register(p);

        Traversal trav = new Traversal () {
                protected void act(Component c) {
                    if (c instanceof Widget) {
                        ((Widget) c).setValidateInvisible(false);
                    }
                }
            };

        trav.preorder(this);
    }

    public void register(Form form, FormModel model) {
        super.register(form, model);
        m_form = form;
    }

    private void addInitialized(String name) {
        m_initialized = new Hidden(new BooleanParameter(name));
        add(m_initialized);
        m_initialized.setDefaultValue(Boolean.FALSE);
    }

    public boolean isInitialized(PageState ps) {
        return m_initialized.getValue(ps).equals(Boolean.TRUE);
    }

    private void setInitialized(PageState ps) {
        m_initialized.setValue(ps, Boolean.TRUE);
    }

    // Turn off forwarding of init events.
    protected FormInitListener createInitListener() {
        return new FormInitListener() {
                public void init(FormSectionEvent evt) { }
            };
    }

    protected void fireSubmitted(FormSectionEvent evt)
        throws FormProcessException {
        if (isInitialized(evt.getPageState())) {
            super.fireSubmitted(evt);
        }
    }

    protected void fireValidate(FormSectionEvent evt) {
        if (isInitialized(evt.getPageState())) {
            super.fireValidate(evt);
        }
    }

    protected void fireProcess(FormSectionEvent evt)
        throws FormProcessException {
        if (isInitialized(evt.getPageState())) {
            super.fireProcess(evt);
        }
    }

    public void generateXML(PageState ps, Element parent) {
        if (!isInitialized(ps)) {
            FormData fd = m_form.getFormData(ps);
            try {
                fireInit(new FormSectionEvent(this, ps, fd));
                setInitialized(ps);
            } catch (FormProcessException ex) {
                s_log.debug("initialization aborted", ex);
                fd.addError("Initialization Aborted: " + ex.getMessages());
            }
        }

        super.generateXML(ps, parent);
    }

}
