/*
 * Copyright (C) 2001-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.bebop;

import java.util.Iterator;

import com.arsdigita.bebop.event.ChangeEvent;
import com.arsdigita.bebop.event.ChangeListener;
import com.arsdigita.bebop.event.EventListenerList;
import com.arsdigita.util.Assert;
import com.arsdigita.util.Lockable;

/**
 * A standard implementation of <code>SingleSelectionModel</code> and <code>Lockable</code>.  Those
 * wishing to define a SingleSelectionModel will ordinarily want to extend
 * this class.
 *
 * @version $Id: //core-platform/dev/src/com/arsdigita/bebop/AbstractSingleSelectionModel.java#6 $
 */
public abstract class AbstractSingleSelectionModel
    implements SingleSelectionModel, Lockable {

    public static final String versionId = "$Id: //core-platform/dev/src/com/arsdigita/bebop/AbstractSingleSelectionModel.java#6 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    private EventListenerList m_listeners;
    private boolean m_locked;

    /** Creates a new AbstractSingleSelectionModel.
     */
    public AbstractSingleSelectionModel() {
        m_listeners = new EventListenerList();
    }

    /**
     * Returns <code>true</code> if there is a selected element.
     *
     * @param state the state of the current request
     * @return <code>true</code> if there is a selected component;
     * <code>false</code> otherwise.
     */
    public boolean isSelected(PageState state) {
        return getSelectedKey(state) != null;
    }

    public abstract Object getSelectedKey(PageState state);

    public abstract void setSelectedKey(PageState state, Object key);

    public void clearSelection(PageState state) {
        setSelectedKey(state, null);
    }

    // Selection change events

    public void addChangeListener(ChangeListener l) {
        Assert.assertNotLocked(this);
        m_listeners.add(ChangeListener.class, l);
    }

    public void removeChangeListener(ChangeListener l) {
        Assert.assertNotLocked(this);
        m_listeners.remove(ChangeListener.class, l);
    }

    protected void fireStateChanged(PageState state) {
        Iterator i = m_listeners.getListenerIterator(ChangeListener.class);
        ChangeEvent e = null;

        while (i.hasNext()) {
            if ( e == null ) {
                e = new ChangeEvent(this, state);
            }
            ((ChangeListener) i.next()).stateChanged(e);
        }
    }

    // implement Lockable
    public void lock() {
        m_locked = true;
    }

    public final boolean isLocked() {
        return m_locked;
    }
}
