/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui;


import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.cms.ui.CMSDHTMLEditor;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.contenttypes.PressRelease;
import com.arsdigita.cms.ui.authoring.BasicPageForm;
import com.arsdigita.cms.util.GlobalizationUtil;

/**
 * Form to edit the basic properties of a press release. These are name, title,
 * release date and reference code. This form can be extended to create forms
 * for PressRelease subclasses.
 **/
public class PressReleasePropertyForm extends BasicPageForm
    implements FormProcessListener, FormInitListener, FormSubmissionListener {

    private PressReleasePropertiesStep m_step;

    /** contact info parameter name */
    public static final String CONTACT_INFO = "contactInfo";
    /**  summary parameter name */
    public static final String SUMMARY = "summary";
    /** Reference code parameter name */
    public static final String REF_CODE = "ref_code";
    /** Release date parameter name */
    public static final String RELEASE_DATE = "rel_date";
    /** Name of this form */
    public static final String ID = "press_release_edit";

    /**
     * Creates a new form to edit the PressRelease object specified
     * by the item selection model passed in.
     * @param itemModel The ItemSelectionModel to use to obtain the
     *    PressRelease to work on
     */
    public PressReleasePropertyForm( ItemSelectionModel itemModel ) {
        this( itemModel, null );
    }

    /**
     * Creates a new form to edit the PressRelease object specified
     * by the item selection model passed in.
     * @param itemModel The ItemSelectionModel to use to obtain the
     *    PressRelease to work on
     * @param step The PressReleasePropertiesStep which controls this form.
     */
    public PressReleasePropertyForm( ItemSelectionModel itemModel, PressReleasePropertiesStep step ) {
        super( ID, itemModel );
        m_step = step;
        addSubmissionListener(this);
    }

    /**
     * Adds widgets to the form.
     */
    protected void addWidgets() {
        super.addWidgets();

        add(new Label(GlobalizationUtil.globalize("cms.contenttypes.ui.contact_info")));
        ParameterModel contactInfoParam = new StringParameter(CONTACT_INFO);
        CMSDHTMLEditor contactInfo = new CMSDHTMLEditor(contactInfoParam);
        contactInfo.setCols(40);
        contactInfo.setRows(10);
        add(contactInfo);

        add(new Label(GlobalizationUtil.globalize("cms.contenttypes.ui.summary")));
        ParameterModel summaryParam = new StringParameter(SUMMARY);
        TextArea summary = new TextArea(summaryParam);
        summary.setCols(40);
        summary.setRows(7);
        add(summary);

        add(new Label(GlobalizationUtil.globalize("cms.contenttypes.ui.reference_code")));
        ParameterModel refCodeParam = new StringParameter(REF_CODE);
        TextField refCode = new TextField(refCodeParam);
        refCode.setSize(30);
        refCode.setMaxLength(30);
        add(refCode);
    }

    /** Form initialisation hook. Fills widgets with data. */
    public void init(FormSectionEvent fse) {
        FormData data = fse.getFormData();
        PressRelease release
            = (PressRelease) super.initBasicWidgets(fse);

        data.put(CONTACT_INFO, release.getContactInfo());
        data.put(SUMMARY,      release.getSummary());
        data.put(REF_CODE,     release.getReferenceCode());
    }

    /** Cancels streamlined editing. */
    public void submitted( FormSectionEvent fse ) {
        if (m_step != null &&
            getSaveCancelSection().getCancelButton()
            .isSelected( fse.getPageState())) {
            m_step.cancelStreamlinedCreation(fse.getPageState());
        }
    }

    /** Form processing hook. Saves PressRelease object. */
    public void process(FormSectionEvent fse) {
        FormData data = fse.getFormData();

        PressRelease release = (PressRelease) super.processBasicWidgets(fse);

        // save only if save button was pressed
        if (release != null
            && getSaveCancelSection().getSaveButton()
            .isSelected(fse.getPageState())) {

            release.setReferenceCode((String) data.get(REF_CODE));
            release.setContactInfo((String) data.get(CONTACT_INFO));
            release.setSummary((String) data.get(SUMMARY));
            release.save();
        }
        if (m_step != null) {
            m_step.maybeForwardToNextStep(fse.getPageState());
        }
    }
}
