/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes;


import com.arsdigita.cms.Article;
import com.arsdigita.cms.ContentType;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.util.Assert;
import java.math.BigDecimal;
import java.text.DateFormat;
import java.util.Date;

/**
 * This content type represents a news item.
 *
 * @version $Revision: #6 $ $Date: 2004/04/07 $
 **/
public class NewsItem extends Article {

    /** PDL property name for lead */
    public static final String LEAD = "lead";
    /** PDL property name for news date */
    public static final String NEWS_DATE = "newsDate";
    public static final String IS_HOMEPAGE = "isHomepage";

    public static final String RECENT_NEWS = 
        "com.arsdigita.cms.contenttypes.RecentNews";

    /** Data object type for this domain object */
    public static final String BASE_DATA_OBJECT_TYPE
        = "com.arsdigita.cms.contenttypes.NewsItem";

    private static final NewsItemConfig s_config = new NewsItemConfig();
    static {
        s_config.load();
    }
    
    public static final NewsItemConfig getConfig() {
        return s_config;
    }

    public NewsItem() {
        this(BASE_DATA_OBJECT_TYPE);
    }

    public NewsItem(BigDecimal id) throws DataObjectNotFoundException {
        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    public NewsItem(OID id) throws DataObjectNotFoundException {
        super(id);
    }

    public NewsItem(DataObject obj) {
        super(obj);
    }

    public NewsItem(String type) {
        super(type);
    }

    public void initialize() {
        super.initialize();
        
        if (isNew()) {
            setIsHomepage(Boolean.FALSE);
        }
    }

    public void beforeSave() {
        super.beforeSave();
        
        Assert.exists(getContentType(), ContentType.class);
    }

    /* accessors *****************************************************/
    public String getLead() {
        return (String) get(LEAD);
    }

    public void setLead(String lead) {
        set(LEAD, lead);
    }

    public Boolean isHomepage() {
        final Boolean isHomepage = (Boolean) get(IS_HOMEPAGE);
        return isHomepage;
    }

    public void setIsHomepage(Boolean isHomePage) {
        set(IS_HOMEPAGE, isHomePage);
    }

    public Date getNewsDate() {
        return (Date) get(NEWS_DATE);
    }

    public String getDisplayNewsDate() {
        Date d = getNewsDate();
        return (d != null) ? DateFormat.getDateInstance(DateFormat.LONG)
            .format(d) : null;
    }

    public void setNewsDate(Date newsDate) {
        set(NEWS_DATE, newsDate);
    }

    public static final int SUMMARY_LENGTH = 200;
    public String getSearchSummary() {
        return com.arsdigita.util.StringUtils.truncateString(getLead(),
                                                             SUMMARY_LENGTH,
                                                             true);
    }

    /*
     * This static method returns the most recent news item (by news_date)
     */
    public static NewsItem getMostRecentNewsItem() {
        DataQuery newsItems = SessionManager.getSession().retrieveQuery(RECENT_NEWS);

        NewsItem newsItem = null;
        if (newsItems.next()) {
            try {
                newsItem = new NewsItem((BigDecimal) newsItems.get("newsID"));
            } catch (DataObjectNotFoundException ex) {
                //
            }
        }
        newsItems.close();

        return newsItem;
    }
}
