/*
 * Copyright (C) 2003-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui.mparticle;

import com.arsdigita.cms.CMS;
import com.arsdigita.cms.CMSContext;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.CMSExcursion;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.contenttypes.MultiPartArticle;
import com.arsdigita.cms.contenttypes.ArticleSection;
import com.arsdigita.cms.contenttypes.ArticleSectionCollection;
import com.arsdigita.cms.dispatcher.XMLGenerator;

import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.SimpleComponent;
import com.arsdigita.bebop.parameters.BigDecimalParameter;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.xml.Element;

import com.arsdigita.domain.DomainObjectXMLRenderer;

import java.io.IOException;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import java.util.ArrayList;

/**
 * <p>This <code>ContentPanel</code> component fetches
 * the {@link com.arsdigita.cms.dispatcher.XMLGenerator} for the content
 * section.</p>
 *
 * @author Michael Pih (pihman@arsdigita.com)
 * @version $Revision: #6 $ $Date: 2004/04/07 $
 */
public class ArticleSectionPanel extends SimpleComponent {

    private static final Logger s_log = Logger.getLogger(ArticleSectionPanel.class);

    private PageParameter m_page;
    private boolean m_showAllSections = false;

    public static final String PAGE_NUMBER_PARAM = "page";

    public static final String versionId = "$Id: //cms/content-types/mparticle/dev/src/com/arsdigita/cms/contenttypes/ui/mparticle/ArticleSectionPanel.java#6 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";

    public ArticleSectionPanel() {
        super();
        
        m_page = new PageParameter(PAGE_NUMBER_PARAM);
    }

    public void register(Page p) {
        super.register(p);
        
        p.addGlobalStateParam(m_page);
    }

    protected XMLGenerator getXMLGenerator(PageState state) {
        ContentSection section = CMS.getContext().getContentSection();
        return section.getXMLGenerator();
    }

    public void setShowAllSections(boolean showAll) {
        m_showAllSections = showAll;
    }

    protected ContentItem getContentItem(PageState state) {
        CMSContext context = CMS.getContext();
        
        if (!context.hasContentItem()) {
            return null;
        }
        return context.getContentItem();
    }

    protected ArticleSection[] getSections(final PageState state) {
        PageNumber number = (PageNumber)state.getValue(m_page);
        if (number == null) {
            number = new PageNumber("1");
        }
        
        MultiPartArticle mpa = (MultiPartArticle)getContentItem(state);
        if (!number.wantAllSections()) {
            if (s_log.isDebugEnabled()) {
                s_log.debug("Page number requested is " + number.getPageNumber());
            }
            return getSections(state, number.getPageNumber());
        } else {
            ArticleSectionCollection sections = mpa.getSections();
            if (s_log.isDebugEnabled()) {
                s_log.debug("No page number provided");
            }
            ArticleSection[] page = new ArticleSection[(int)sections.size()];
            int i = 0;
            while (sections.next()) {
                page[i] = (ArticleSection)sections.getArticleSection();
                i++;
            }
            return page;
        }
    }
    
    // Get the section based on position in list of sections
    protected ArticleSection[] getSections(PageState state,
                                           Integer number) {
        if (s_log.isDebugEnabled()) {
            s_log.debug("Getting page " + number);
        }

        MultiPartArticle mpa = (MultiPartArticle)getContentItem(state);
        ArticleSectionCollection sections = mpa.getSections();
        int current = 1;
        int desired = number.intValue();

        ArrayList page;
        try {
            // Skip over sections until we hit the desired page
            while (current < desired) {
                if (!sections.next()) {
                    if (s_log.isDebugEnabled()) {
                        s_log.debug("Hit end of section list");
                    }
                    return new ArticleSection[] {};
                }
                ArticleSection section = (ArticleSection)sections.getArticleSection();
                if (s_log.isDebugEnabled()) {
                    s_log.debug("Skipping " + section.getOID());
                }
                if (section.isPageBreak()) {
                    if (s_log.isDebugEnabled()) {
                        s_log.debug("Got page break");
                    }
                    current++;
                }
            }

            // Now capture sections until we hit the next page (or end of sections)
            int subsequent = desired + 1;
            page = new ArrayList();
            while (current < subsequent) {
                if (!sections.next()) {
                    if (s_log.isDebugEnabled()) {
                        s_log.debug("Got end of pages, returning.");
                    }
                    break;
                }
                ArticleSection section = (ArticleSection)sections.getArticleSection();
                page.add(section);
                if (s_log.isDebugEnabled()) {
                    s_log.debug("Keeping section " + section.getOID());
                }
                if (section.isPageBreak()) {
                    if (s_log.isDebugEnabled()) {
                        s_log.debug("Got page break");
                    }
                    current++;
                }
            }
        } finally {
            sections.close();
        }

        if (s_log.isDebugEnabled()) {
            s_log.debug("All done " + page.size() + " sections found");
        }
        
        return (ArticleSection[])page.toArray(new ArticleSection[page.size()]);
    }

    /**
     * Generates XML that represents a content item.
     *
     * @param state The page state
     * @param parent The parent DOM element
     * @see com.arsdigita.cms.dispatcher.XMLGenerator
     */
    public void generateXML(final PageState state, 
                            final Element parent) {
        ContentItem item = getContentItem(state);

        if (!isVisible(state) || item == null ||
            !(item instanceof MultiPartArticle)) {
            if (s_log.isDebugEnabled()) {
                s_log.debug("Skipping generate XML isVisible: " + 
                            isVisible(state) +
                            " item " + 
                            (item == null ? null : item.getOID()));
            }
            return;
        }

        Element content = parent.newChildElement(
            "cms:articleSectionPanel", CMS.CMS_XML_NS);
        exportAttributes(content);
        
        XMLGenerator xmlGenerator = getXMLGenerator(state);
        
        ArticleSection sections[] = getSections(state);
        for (int i= 0 ; i < sections.length ; i++) {
            generateSectionXML(state, 
                               content, 
                               sections[i],
                               xmlGenerator);
        }
    }
                                           
    
    protected void generateSectionXML(final PageState state,
                                      final Element parent,
                                      final ContentItem section,
                                      final XMLGenerator xmlGenerator) {
        CMSExcursion excursion = new CMSExcursion() {
                public void excurse() {
                    setContentItem(section);
                    
                    xmlGenerator.generateXML(state, parent, null);
                }
            };
        try {
            excursion.run();
        } catch (ServletException ex) {
            throw new UncheckedWrapperException("excursion failed", ex);
        } catch (IOException ex) {
            throw new UncheckedWrapperException("excursion failed", ex);
        }
    }

    // A class representing either an Integer number indicating
    // the position in the list of sections, or the string 'all'
    private class PageNumber {
        
        private boolean m_all;
        private Integer m_number;
        
        public PageNumber(String number) 
            throws NumberFormatException {
            
            if ("all".equals(number.toLowerCase())) {
                m_all = true;
                m_number = null;
            } else {
                m_all = false;
                m_number = new Integer(number);
            }
        }
        
        public boolean wantAllSections() {
            return m_all;
        }
        
        public Integer getPageNumber() {
            return m_number;
        }
    }
    
    // A parameter which is either an Integer number indicating
    // the position in the list of sections, or the string 'all'
    private class PageParameter extends ParameterModel {
        
        public static final String versionId = "$Id: //cms/content-types/mparticle/dev/src/com/arsdigita/cms/contenttypes/ui/mparticle/ArticleSectionPanel.java#6 $ by $Author: dennis $, $DateTime: 2004/04/07 16:07:11 $";
        
        public PageParameter(String name) {
            super(name);
        }
        
        public Object transformValue(HttpServletRequest request)
            throws IllegalArgumentException {
            return transformSingleValue(request);
        }
        
        public Object unmarshal(String encoded)
            throws IllegalArgumentException {
            
            if (encoded == null || encoded.length() == 0) {
                return null;
            }
            try {
                return new PageNumber(encoded);
            } catch (NumberFormatException e) {
                e.printStackTrace();
                throw new IllegalArgumentException
                    (getName() + " should be a BigDecimal: '" + encoded + "'");
            }
        }
        
        public Class getValueClass() {
            return PageNumber.class;
        }
        
    }

}
