/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.ui.formbuilder;


import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.formbuilder.FormSectionItem;
import com.arsdigita.toolbox.ui.DomainObjectPropertySheet;
import com.arsdigita.cms.ui.authoring.AuthoringKitWizard;
import com.arsdigita.cms.ui.authoring.BasicPageForm;
import com.arsdigita.cms.ui.authoring.SimpleEditStep;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.formbuilder.PersistentFormSection;
import com.arsdigita.cms.ui.workflow.WorkflowLockedComponentAccess;


public class FormSectionProperties extends SimpleEditStep {

    /** The name of the editing sheet added to this step */
    public static String EDIT_SHEET_NAME = "edit";

    public FormSectionProperties(ItemSelectionModel model,
                                 AuthoringKitWizard parent) {
        super(model, parent);

        setDefaultEditKey(EDIT_SHEET_NAME);
        BasicPageForm edit = new FormSectionPropertyEditForm(model, this);
        add(EDIT_SHEET_NAME, "Edit", 
            new WorkflowLockedComponentAccess(edit, model),
            edit.getSaveCancelSection().getCancelButton());
        setDisplayComponent(buildDisplayComponent(model));
    }


    protected class FormSectionPropertyEditForm extends BasicPageForm
        implements FormProcessListener, FormInitListener, FormSubmissionListener {

        private FormSectionProperties m_step;
        private TextArea m_desc;

        public FormSectionPropertyEditForm(ItemSelectionModel itemModel) {
            this(itemModel, null);
        }

        /**
         *  @param itemModel the ItemSelectionModel that controls which form
         *                   to work on
         *  @param sectionProperties The properties step that controls this form.
         */
        public FormSectionPropertyEditForm(ItemSelectionModel itemModel,
                                           FormSectionProperties sectionProperties) {
            super("edit_properties", itemModel);
            m_step = sectionProperties;
            addSubmissionListener(this);
        }

        protected void addWidgets() {
            super.addWidgets();

            m_desc = new TextArea(new StringParameter("description"));
            m_desc.setRows(5);
            m_desc.setCols(50);
            add(new Label(GlobalizationUtil.globalize("cms.ui.formbuilder.description")));
            add(m_desc);
        }

        public void init(FormSectionEvent e)
            throws FormProcessException {

            FormSectionItem item = (FormSectionItem)initBasicWidgets(e);

            PersistentFormSection form = item.getFormSection();
            m_desc.setValue(e.getPageState(), form.getDescription());
        }

        public void process(FormSectionEvent e)
            throws FormProcessException {

            FormData data = e.getFormData();
            FormSectionItem item = (FormSectionItem)processBasicWidgets(e);

            PersistentFormSection form = item.getFormSection();

            item.save();
            form.setAdminName(item.getName());
            form.setDescription((String)data.get("description"));
            form.save();

            if (m_step != null) {
                m_step.maybeForwardToNextStep(e.getPageState());
            }
        }

        /** Cancels streamlined editing. */
        public void submitted( FormSectionEvent fse ) {
            if (m_step != null &&
                getSaveCancelSection().getCancelButton()
                .isSelected( fse.getPageState())) {
                m_step.cancelStreamlinedCreation(fse.getPageState());
            }
        }
    }

    private Component buildDisplayComponent(ItemSelectionModel itemModel) {
        DomainObjectPropertySheet sheet = new DomainObjectPropertySheet(itemModel);

        sheet.add(GlobalizationUtil.globalize("cms.ui.formbuilder.name"),  
                  ContentPage.NAME);
        sheet.add(GlobalizationUtil.globalize("cms.ui.formbuilder.title"),  
                  ContentPage.TITLE);
        sheet.add(GlobalizationUtil.globalize("cms.ui.formbuilder.description"),
                  "form", new FormSectionFormatter());

        return sheet;
    }

    private class FormSectionFormatter implements DomainObjectPropertySheet.AttributeFormatter {
        ItemSelectionModel m_item;

        public String format(DomainObject item, String attribute, PageState state) {
            if (attribute.equals("form")) {
                FormSectionItem section = (FormSectionItem)item;

                PersistentFormSection s = section.getFormSection();
                return s.getDescription();
            }
            return null;
        }

    }
}
