/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.formbuilder;

import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.Form;
import com.arsdigita.bebop.FormSection;
import com.arsdigita.bebop.Page;
import com.arsdigita.bebop.PageState;
import com.arsdigita.bebop.form.Widget;
import com.arsdigita.bebop.util.Traversal;
import com.arsdigita.bebop.FormData;
import com.arsdigita.cms.CMS;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.CustomCopy;
import com.arsdigita.cms.ItemCopier;
import com.arsdigita.cms.dispatcher.XMLGenerator;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.formbuilder.PersistentComponentFactory;
import com.arsdigita.formbuilder.PersistentFormSection;
import com.arsdigita.formbuilder.ui.BaseAddObserver;
import com.arsdigita.formbuilder.ui.PlaceholdersInitListener;
import com.arsdigita.formbuilder.ui.FormBuilderXMLRenderer;
import com.arsdigita.formbuilder.util.FormBuilderUtil;
import com.arsdigita.web.Web;
import com.arsdigita.web.URL;
import com.arsdigita.globalization.GlobalizedMessage;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.metadata.Property;
import com.arsdigita.persistence.OID;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.xml.Element;
import com.arsdigita.domain.DomainObjectXMLRenderer;
import com.arsdigita.cms.dispatcher.SimpleXMLGenerator;
import com.arsdigita.formbuilder.ui.ComponentTraverse;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Iterator;


public class FormSectionItem extends ContentPage
    implements XMLGenerator {

    public static final String BASE_DATA_OBJECT_TYPE
        = "com.arsdigita.cms.formbuilder.FormSectionItem";

    public static final String FORM_SECTION = "formSection";

    public FormSectionItem() {
        this(BASE_DATA_OBJECT_TYPE);
    }

    public FormSectionItem(String typeName) {
        super(typeName);
    }

    // Not with content page we can't :(
    /*
      public FormSectionItem(ObjectType type) {
      super(type);
      }
    */

    public FormSectionItem(DataObject obj) {
        super(obj);
    }

    public FormSectionItem(BigDecimal id)
        throws DataObjectNotFoundException {

        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    public FormSectionItem(OID oid)
        throws DataObjectNotFoundException {

        super(oid);
    }

    protected void beforeSave() {
        if (isNew() && get(FORM_SECTION) == null) {
            PersistentFormSection formSection = new PersistentFormSection();
            formSection.setAdminName(getName());
            formSection.save();
            setAssociation(FORM_SECTION, formSection);
        }

        super.beforeSave();
    }


    public void delete() {
        PersistentFormSection form = getFormSection();

        Collection children = form.getComponents();
        form.clearComponents();

        Iterator i = children.iterator();
        while (i.hasNext()) {
            PersistentComponentFactory c = (PersistentComponentFactory)i.next();
            c.delete();
        }

        super.delete();

        form.delete();
    }

    public boolean copyProperty(CustomCopy src,
                                Property property,
                                ItemCopier copier) {
        if (property.getName().equals(FORM_SECTION)) {
            setAssociation(
                FORM_SECTION, (new FormCopier())
                .copyFormSection(((FormSectionItem)src).getFormSection()));
            return true;
        }

        return super.copyProperty(src, property, copier);
    }


    public String getBaseDataObjectType() {
        return BASE_DATA_OBJECT_TYPE;
    }

    public PersistentFormSection getFormSection() {
        return new PersistentFormSection((DataObject)get(FORM_SECTION));
    }


    public void generateXML(PageState state,
                            Element parent,
                            String useContext) {
        PersistentFormSection form = getFormSection();
        form.setComponentAddObserver( new BaseAddObserver());

        Form c = new Form("formSectionItem");
        c.add((FormSection)form.createComponent());

        c.addInitListener(new PlaceholdersInitListener());

        // Make the form readonly
        Traversal t = new Traversal() {
                public void act(Component c) {
                    try {
                        Widget widget = (Widget)c;
                        widget.setDisabled();
                        widget.setReadOnly();
                    } catch (ClassCastException ex) {
                        // Nada
                    }
                }
            };
        t.preorder(c);

        // Fake the page context for the item, since we
        // have no access to the real page context.
        Page p = new Page("dummy");
        p.add(c);
        p.lock();

        PageState fake;
        try {
            fake = p.process(new NoParametersHttpServletRequest(
                                 state.getRequest()), state.getResponse());
        } catch (Exception e) {
            throw new UncheckedWrapperException(e);
        }

        Traversal t2 = new VisibleTraverse(fake);
        t2.preorder(c);

        // Simply embed the bebop xml as a child of the cms:item tag
        Element element = parent.newChildElement("cms:item", CMS.CMS_XML_NS);

        String action = c.getAction();
        if (action == null) {
            final URL requestURL = Web.getContext().getRequestURL();

            if (requestURL == null) {
                action = state.getRequest().getRequestURI();
            } else {
                action = requestURL.getRequestURI();
            }
        }

        element.addAttribute(FormBuilderUtil.FORM_ACTION, action);
        
        FormBuilderXMLRenderer renderer = 
            new FormBuilderXMLRenderer(element);
        
        renderer.setWrapAttributes(true);
        renderer.setWrapRoot(false);
        renderer.setWrapObjects(false);
        
        renderer.walk(this, SimpleXMLGenerator.ADAPTER_CONTEXT);

        // we need to generate the state so that it can be part of the form
        // and correctly included when the form is submitted.  We could
        // do this by iterating through the form data but it does not
        // seem like a good idea to just cut and paste the code out
        // of the PageState class
        state.generateXML(element.newChildElement
                          ("cms:pageState", CMS.CMS_XML_NS));

        // then, if the component is actually a form, we need
        // to print out any possible errors
        // Ideally we could do this as part of the "walk" but for now
        // that does not work because we don't pass in the page state
        // although that can always we updated.
        if (c instanceof Form) {
            Element infoElement = 
                element.newChildElement(FormBuilderUtil.FORMBUILDER_FORM_INFO,
                                        FormBuilderUtil.FORMBUILDER_XML_NS);
            Form f = (Form)c;

            Traversal infoTraversal = 
                new ComponentTraverse(state, ((Form)c).getFormData(state), 
                                      infoElement);
            infoTraversal.preorder(f);
        }

        // we need to generate the state so that it can be part of the form
        // and correctly included when the form is submitted.  We could
        // do this by iterating through the form data but it does not
        // seem like a good idea to just cut and paste the code out
        // of the PageState class
        fake.setControlEvent(c);    
        fake.generateXML(element.newChildElement
                         (FormBuilderUtil.FORMBUILDER_PAGE_STATE, 
                          FormBuilderUtil.FORMBUILDER_XML_NS));
    }


    private class VisibleTraverse extends Traversal {

        PageState m_state;
        VisibleTraverse(PageState state) {
            m_state = state;
        }
        public void act(Component c) {
            try {
                m_state.setVisible(c, true);
            } catch (ClassCastException ex) {
                // Nada
            }
        }
    }
}
