/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui;

import com.arsdigita.cms.contenttypes.FAQItem;
import com.arsdigita.cms.contenttypes.util.FAQGlobalizationUtil;
import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.ui.authoring.BasicPageForm;


/**
 * Form to edit the basic properties of an FAQItem. This form can be
 * extended to create forms for FAQItem subclasses.
 */
public class FAQItemPropertyForm
    extends BasicPageForm
    implements FormProcessListener, FormInitListener,FormSubmissionListener {

    private FAQItemPropertiesStep m_step;

    /** parameter names */
    public static final String QUESTION = "question";
    public static final String ANSWER = "answer";
    public static final String SECTION_NAME = "sectionName";

    /** Name of this form */
    public static final String ID = "FAQItem_edit";

    /**
     * Creates a new form to edit the FAQItem object specified by the item
     * selection model passed in.
     *
     * @param itemModel The ItemSelectionModel to use to obtain the FAQItem to
     * work on
     **/
    public FAQItemPropertyForm( ItemSelectionModel itemModel ) {
        this(itemModel,null);
    }
    /**
     * Creates a new form to edit the FAQItem object specified by the item
     * selection model passed in.
     *
     * @param itemModel The ItemSelectionModel to use to obtain the FAQItem to
     * work on
     * @param step The FAQItemPropertiesStep which controls this form.
     **/
    public FAQItemPropertyForm( ItemSelectionModel itemModel, FAQItemPropertiesStep step ) {
        super( ID, itemModel );
        m_step = step;
        addSubmissionListener(this);
    }

    /**
     * Adds widgets to the form.
     */
    protected void addWidgets() {
        super.addWidgets();

        add( new Label(FAQGlobalizationUtil.globalize("cms.contenttypes.ui.faq.question")) );
        ParameterModel questionParam
            = new StringParameter( QUESTION );
        questionParam
            .addParameterListener( new NotNullValidationListener() );
        TextArea question = new TextArea( questionParam );
        question.setCols( 40 );
        question.setRows( 5 );
        add( question );

        add( new Label(FAQGlobalizationUtil.globalize("cms.contenttypes.ui.faq.answer")) );
        ParameterModel answerParam = new StringParameter( ANSWER );
        answerParam
            .addParameterListener( new NotNullValidationListener() );
        TextArea answer = new TextArea( answerParam );
        answer.setCols( 40 );
        answer.setRows( 5 );
        add( answer );
        
        add( new Label(FAQGlobalizationUtil.globalize("cms.contenttypes.ui.faq.sectionName")) );
        ParameterModel sectionNameParam = new StringParameter( SECTION_NAME );
        TextField sectionName = new TextField(sectionNameParam); 
        add(sectionName);

    }

    /** Form initialisation hook. Fills widgets with data. */
    public void init( FormSectionEvent fse ) {
        FormData data = fse.getFormData();
        FAQItem faqItem = (FAQItem) super.initBasicWidgets( fse );

        data.put( QUESTION, faqItem.getQuestion() );
        data.put( ANSWER,   faqItem.getAnswer() );
        data.put(SECTION_NAME , faqItem.getSectionName());
    }

    /** Cancels streamlined editing. */
    public void submitted( FormSectionEvent fse ) {
        if (m_step != null &&
            getSaveCancelSection().getCancelButton()
            .isSelected( fse.getPageState())) {
            m_step.cancelStreamlinedCreation(fse.getPageState());
        }
    }
 
    /** Form processing hook. Saves FAQItem object. */
    public void process( FormSectionEvent fse ) {
        FormData data = fse.getFormData();
        
        FAQItem faqItem = (FAQItem) super.processBasicWidgets( fse );

        // save only if save button was pressed
        if( faqItem != null
            && getSaveCancelSection().getSaveButton()
            .isSelected( fse.getPageState() ) ) {
          faqItem.setQuestion( (String) data.get( QUESTION ) );
          faqItem.setAnswer( (String) data.get( ANSWER ) );
          faqItem.setSectionName( (String) data.get( SECTION_NAME) );
          faqItem.save();
        }
        if (m_step != null) {
            m_step.maybeForwardToNextStep(fse.getPageState());
        }
    }
}
