/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes;


import com.arsdigita.cms.ContentType;
import com.arsdigita.cms.TextPage;
import com.arsdigita.cms.util.GlobalizationUtil;
import com.arsdigita.domain.DataObjectNotFoundException;
import com.arsdigita.persistence.DataObject;
import com.arsdigita.persistence.DataQuery;
import com.arsdigita.persistence.OID;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.util.UncheckedWrapperException;
import com.arsdigita.util.Assert;

import java.math.BigDecimal;
import java.text.DateFormat;
import java.util.Date;

/**
 * This content type represents an event.
 *
 * @version $Revision: #5 $ $Date: 2004/04/07 $
 **/
public class Event extends TextPage {

    private final static org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(Event.class);

    /** PDL property name for event date */
    public static final String START_DATE = "startDate";
    public static final String END_DATE = "endDate";

    public static final String START_TIME = "startTime";
    public static final String END_TIME = "endTime";

    public static final String EVENT_DATE = "eventDate";
    /** PDL property name for location */
    public static final String LOCATION = "location";
    /** PDL property name for lead */
    public static final String LEAD = "lead";
    /** PDL property name for main contributor */
    public static final String MAIN_CONTRIBUTOR = "mainContributor";
    /** PDL property name for event type */
    public static final String EVENT_TYPE = "eventType";
    /** PDL property name for map link */
    public static final String MAP_LINK = "mapLink";
    /** PDL property name for cost */
    public static final String COST = "cost";

    public static final String RECENT_EVENT = "com.arsdigita.cms.contenttypes.RecentEvent";

    /** Data object type for this domain object */
    public static final String BASE_DATA_OBJECT_TYPE
        = "com.arsdigita.cms.contenttypes.Event";

    public Event() {
        this(BASE_DATA_OBJECT_TYPE);
    }

    public Event(BigDecimal id) throws DataObjectNotFoundException {
        this(new OID(BASE_DATA_OBJECT_TYPE, id));
    }

    public Event(OID id) throws DataObjectNotFoundException {
        super(id);
    }

    public Event(DataObject obj) {
        super(obj);
    }

    public Event(String type) {
        super(type);
    }

    public void beforeSave() {
        super.beforeSave();
        
        Assert.exists(getContentType(), ContentType.class);
    }

    public String formatDate (Date date) {
        return (date != null) ? DateFormat.getDateInstance(DateFormat.LONG)
            .format(date) : null;
    }


    public String formatTime (Date time) {
        return (time != null) ? DateFormat.getTimeInstance(DateFormat.SHORT)
            .format(time) : null;
    }

    /* accessors *****************************************************/
    public Date getStartTime () {
        return (Date)get(START_TIME);
    }

    public void setStartTime (Date startTime) {
        set(START_TIME, startTime);
    }

    public String getDisplayStartTime () {
        return formatTime(getStartTime());
    }

    public Date getEndTime () {
        return (Date)get(END_TIME);
    }

    public void setEndTime (Date endTime) {
        set(END_TIME, endTime);
    }

    public String getDisplayEndTime () {
        return formatTime(getEndTime());
    }

    public Date getStartDate() {
        Date startDate = (Date) get (START_DATE);
        return startDate;
    }

    public void setStartDate(Date startDate) {
        set(START_DATE, startDate);
    }

    public String getDisplayStartDate() {
        return formatDate(getStartDate());
    }

    public Date getEndDate() {
        Date endDate = (Date) get (END_DATE);
        return endDate;
    }

    public String getDisplayEndDate() {
        return formatDate(getEndDate());
    }

    public void setEndDate(Date endDate) {
        set(END_DATE, endDate);
    }

    public String getEventDate() {
        return (String) get(EVENT_DATE);
    }


    public void setEventDate(String eventDate) {
        set(EVENT_DATE, eventDate);
    }

    public String getLocation() {
        return (String) get(LOCATION);
    }

    public void setLocation(String location) {
        set(LOCATION, location);
    }

    public String getLead() {
        return (String) get(LEAD);
    }

    public void setLead(String lead) {
        set(LEAD, lead);
    }

    public String getMainContributor() {
        return (String) get(MAIN_CONTRIBUTOR);
    }

    public void setMainContributor(String mainContributor) {
        set(MAIN_CONTRIBUTOR, mainContributor);
    }

    public String getEventType() {
        return (String) get(EVENT_TYPE);
    }

    public void setEventType(String eventType) {
        set(EVENT_TYPE, eventType);
    }

    public String getMapLink() {
        return (String) get(MAP_LINK);
    }

    public void setMapLink(String mapLink) {
        set(MAP_LINK, mapLink);
    }

    public String getCost() {
        return (String) get(COST);
    }

    public void setCost(String cost) {
        set(COST, cost);
    }

    public static final int SUMMARY_LENGTH = 200;
    public String getSearchSummary() {
        return com.arsdigita.util.StringUtils.truncateString(getLead(),
                                                             SUMMARY_LENGTH,
                                                             true);
    }


    /*
     * Returns a random event which is not over
     */
    public static Event getRandomEvent() {
        DataQuery events = SessionManager.getSession().retrieveQuery(RECENT_EVENT);

        Event event = null;
        long size = events.size();

        if (size > 0) {
            events.rewind();

            long n = (System.currentTimeMillis() % size) + 1;
            int count = 0;

            while (n > count) {
                count++;
                events.next();
                if (n == count) {
                    try {
                        event = new Event((BigDecimal) events.get("eventID"));
                    } catch (DataObjectNotFoundException ex) {
                        //
                    }
                }
            }
        }
        events.close();

        return event;
    }
}
