/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui;


import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.PageState;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.contenttypes.Event;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.toolbox.ui.DomainObjectPropertySheet;
import com.arsdigita.cms.ui.authoring.AuthoringKitWizard;
import com.arsdigita.cms.ui.authoring.BasicPageForm;
import com.arsdigita.cms.ui.authoring.SimpleEditStep;
import com.arsdigita.cms.ui.workflow.WorkflowLockedComponentAccess;
import com.arsdigita.cms.contenttypes.util.EventGlobalizationUtil;

import java.text.DateFormat;

/**
 * Authoring step to edit the simple attributes of the Event content type (and
 * its subclasses). The attributes edited are 'name', 'title', 'event date',
 * 'location', 'lead', 'main contributor', 'event type', 'map link', and
 * 'cost'. This authoring step replaces the
 * <code>com.arsdigita.ui.authoring.PageEdit</code> step for this type.
 **/
public class EventPropertiesStep extends SimpleEditStep {

    /** The name of the editing sheet added to this step */
    public static String EDIT_SHEET_NAME = "edit";

    public EventPropertiesStep(ItemSelectionModel itemModel,
                               AuthoringKitWizard parent) {
        super(itemModel, parent);

        setDefaultEditKey(EDIT_SHEET_NAME);
        BasicPageForm editSheet;

        editSheet = new EventPropertyForm(itemModel, this);
        add(EDIT_SHEET_NAME, "Edit", new WorkflowLockedComponentAccess(editSheet, itemModel),
            editSheet.getSaveCancelSection().getCancelButton());

        setDisplayComponent(getEventPropertySheet(itemModel));
    }

    /**
     * Returns a component that displays the properties of the Event specified
     * by the ItemSelectionModel passed in.
     *
     * @param itemModel The ItemSelectionModel to use
     * @pre itemModel != null
     * @return A component to display the state of the basic properties
     *  of the release
     **/
    public static Component getEventPropertySheet(ItemSelectionModel
                                                  itemModel) {
        DomainObjectPropertySheet sheet = new DomainObjectPropertySheet(itemModel);

        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.name"),  Event.NAME);
        sheet.add(EventGlobalizationUtil.globalize("cms.contenttypes.ui.title"),  Event.TITLE);
        if (!ContentSection.getConfig().getHideLaunchDate()) {
            sheet.add(EventGlobalizationUtil.globalize("cms.contenttypes.ui.launch_date"),
                      ContentPage.LAUNCH_DATE,
                      new DomainObjectPropertySheet.AttributeFormatter() {
                          public String format(DomainObject item,
                                               String attribute,
                                               PageState state) {
                              ContentPage page = (ContentPage) item;
                              if(page.getLaunchDate() != null) {
                                  return DateFormat.getDateInstance(DateFormat.LONG)
                                      .format(page.getLaunchDate());
                              } else {
                                  return "<i>unknown</i>";
                              }
                          }
                      });
        }
        sheet.add(EventGlobalizationUtil.globalize("cms.contenttypes.ui.start_time"),
                  Event.START_DATE,
                  new DomainObjectPropertySheet.AttributeFormatter() {
                      public String format(DomainObject item,
                                           String attribute,
                                           PageState state) {
                          Event e = (Event) item;

                          if (e.getStartDate() != null) {
                              StringBuffer buf = new StringBuffer();
                              buf.append(DateFormat.getDateInstance(DateFormat.LONG)
                                         .format(e.getStartDate()));

                              if (e.getStartTime() != null) {
                                  buf.append(". " + e.getDisplayStartTime());
                              }

                              return buf.toString();
                          } else {
                              return "<i>unknown</i>";
                          }
                      }
                  });

        sheet.add(EventGlobalizationUtil.globalize("cms.contenttypes.ui.end_time"),
                  Event.END_DATE,
                  new DomainObjectPropertySheet.AttributeFormatter() {
                      public String format(DomainObject item,
                                           String attribute,
                                           PageState state) {
                          Event e = (Event) item;
                          if (e.getEndDate() != null) {
                              StringBuffer buf = new StringBuffer();
                              buf.append(DateFormat.getDateInstance(DateFormat.LONG)
                                         .format(e.getEndDate()));

                              if (e.getEndTime() != null) {
                                  buf.append(". " + e.getDisplayEndTime());
                              }

                              return buf.toString();
                          } else {
                              return "<i>unknown</i>";
                          }
                      }
                  });

        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.date_description"),  Event.EVENT_DATE);
        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.location"),  Event.LOCATION);
        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.lead"),  Event.LEAD);
        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.main_contributor"),  Event.MAIN_CONTRIBUTOR);
        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.event_type"),  Event.EVENT_TYPE);
        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.link_to_map"),  Event.MAP_LINK);
        sheet.add( EventGlobalizationUtil.globalize("cms.contenttypes.ui.cost"),  Event.COST);

        return sheet;
    }
}
