/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui;


import com.arsdigita.bebop.FormData;
import com.arsdigita.bebop.FormProcessException;
import com.arsdigita.bebop.Label;
import com.arsdigita.bebop.event.FormInitListener;
import com.arsdigita.bebop.event.FormProcessListener;
import com.arsdigita.bebop.event.FormSectionEvent;
import com.arsdigita.bebop.event.FormSubmissionListener;
import com.arsdigita.bebop.form.TextArea;
import com.arsdigita.bebop.form.TextField;
import com.arsdigita.bebop.form.Time;
import com.arsdigita.bebop.parameters.DateParameter;
import com.arsdigita.bebop.parameters.NotNullValidationListener;
import com.arsdigita.bebop.parameters.ParameterModel;
import com.arsdigita.bebop.parameters.StringParameter;
import com.arsdigita.bebop.parameters.TimeParameter;
import com.arsdigita.bebop.parameters.TrimmedStringParameter;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.contenttypes.Event;
import com.arsdigita.cms.ui.authoring.BasicPageForm;
import com.arsdigita.cms.contenttypes.util.EventGlobalizationUtil;
import com.arsdigita.cms.ui.CMSDHTMLEditor;

/**
 * Form to edit the basic properties of an event. This form can be extended to
 * create forms for Event subclasses.
 **/
public class EventPropertyForm extends BasicPageForm
    implements FormProcessListener, FormInitListener, FormSubmissionListener {

    private final static org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(EventPropertyForm.class);


    private EventPropertiesStep m_step;

    /** event date parameter name */
    public static final String START_DATE = "startDate";
    public static final String END_DATE = "endDate";
    public static final String EVENT_DATE = "eventDate";

    public static final String START_TIME = "startTime";
    public static final String END_TIME    = "endTime";

    /**  location parameter name */
    public static final String LOCATION = "location";
    /**  lead parameter name */
    public static final String LEAD = "lead";
    /** Main contributor parameter name */
    public static final String MAIN_CONTRIBUTOR = "main_contributor";
    /** Event type parameter name */
    public static final String EVENT_TYPE = "event_type";
    /** Map link parameter name */
    public static final String MAP_LINK = "map_link";
    /** cost parameter name */
    public static final String COST = "cost";
    /** Name of this form */
    public static final String ID = "event_edit";

    /**
     * Creates a new form to edit the Event object specified by the item
     * selection model passed in.
     *
     * @param itemModel The ItemSelectionModel to use to obtain the Event to
     * work on
     **/
    public EventPropertyForm( ItemSelectionModel itemModel ) {
        this(itemModel,null);
    }
    /**
     * Creates a new form to edit the Event object specified by the item
     * selection model passed in.
     *
     * @param itemModel The ItemSelectionModel to use to obtain the Event to
     * work on
     * @param step The EventPropertiesStep which controls this form.
     **/
    public EventPropertyForm( ItemSelectionModel itemModel, EventPropertiesStep step ) {
        super( ID, itemModel );
        m_step = step;
        addSubmissionListener(this);
    }

    /**
     * Adds widgets to the form.
     **/
    protected void addWidgets() {
        super.addWidgets();

        /* Start and end dates */
        ParameterModel eventStartDateParam = new DateParameter(START_DATE);
        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.start_date")));
        eventStartDateParam.addParameterListener(new NotNullValidationListener());
        // Use bebop date instead of java.util.date
        com.arsdigita.bebop.form.Date startDate
            = new com.arsdigita.bebop.form.Date(eventStartDateParam);
        add(startDate);

        ParameterModel eventEndDateParam
            = new DateParameter(END_DATE);
        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.end_date")));

        // Use bebop date instead of java.util.date
        com.arsdigita.bebop.form.Date endDate
            = new com.arsdigita.bebop.form.Date(eventEndDateParam);
        add(endDate);


        ParameterModel eventStartTimeParam = new TimeParameter(START_TIME);
        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.start_time")));
        eventStartTimeParam.addParameterListener(new NotNullValidationListener());
        Time startTime = new Time(eventStartTimeParam);
        add(startTime);

        ParameterModel eventEndTimeParam = new TimeParameter(END_TIME);
        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.end_time")));
        Time endTime = new Time(eventEndTimeParam);
        add(endTime);


        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.date_description")));
        ParameterModel eventDateParam = new StringParameter(EVENT_DATE);
        //eventDateParam
        //    .addParameterListener(new NotNullValidationListener());
        CMSDHTMLEditor eventDate = new CMSDHTMLEditor(eventDateParam);
        eventDate.setCols(40);
        eventDate.setRows(8);
        add(eventDate);

        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.location")));
        ParameterModel locationParam = new StringParameter(LOCATION);
        //locationParam
        //    .addParameterListener(new NotNullValidationListener());
        CMSDHTMLEditor location = new CMSDHTMLEditor(locationParam);
        location.setCols(40);
        location.setRows(8);
        add(location);

        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.lead")));
        ParameterModel leadParam = new StringParameter(LEAD);
        //leadParam
        //    .addParameterListener(new NotNullValidationListener());
        TextArea lead = new TextArea(leadParam);
        lead.setCols(40);
        lead.setRows(5);
        add(lead);

        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.main_contributor")));
        ParameterModel mainContributorParam =
            new StringParameter(MAIN_CONTRIBUTOR);
        //mainContributorParam
        //    .addParameterListener(new NotNullValidationListener());
        CMSDHTMLEditor mainContributor = new CMSDHTMLEditor(mainContributorParam);
        mainContributor.setCols(40);
        mainContributor.setRows(10);
        add(mainContributor);

        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.event_type")));
        ParameterModel eventTypeParam = new StringParameter(EVENT_TYPE);
        //eventTypeParam
        //    .addParameterListener(new NotNullValidationListener());
        TextField eventType = new TextField(eventTypeParam);
        eventType.setSize(30);
        eventType.setMaxLength(30);
        add(eventType);

        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.link_to_map")));
        ParameterModel mapLinkParam = new StringParameter(MAP_LINK);
        //mapLinkParam
        //    .addParameterListener(new NotNullValidationListener());
        TextArea mapLink = new TextArea(mapLinkParam);
        mapLink.setCols(40);
        mapLink.setRows(2);
        add(mapLink);

        add(new Label(EventGlobalizationUtil.globalize("cms.contenttypes.ui.cost")));
        ParameterModel costParam = new TrimmedStringParameter(COST);
        TextField cost = new TextField(costParam);
        cost.setSize(30);
        cost.setMaxLength(30);
        add(cost);

    }

    public void validate(FormSectionEvent e) throws FormProcessException {
        FormData d = e.getFormData();
        java.util.Date startDate = d.getDate(START_DATE);
        java.util.Date endDate = d.getDate(END_DATE);

        if (endDate != null) {

            if (startDate == null || startDate.compareTo(endDate) > 0) {
                throw new FormProcessException((String)EventGlobalizationUtil.globalize("cms.contenttypes.ui.event.end_date_after_start_date").localize());
            }
        }
    }




    /** Form initialisation hook. Fills widgets with data. */
    public void init(FormSectionEvent fse) {
        // Do some initialization hook stuff
        FormData data = fse.getFormData();
        Event event
            = (Event) super.initBasicWidgets(fse);

        java.util.Date startDate = event.getStartDate();
        if (startDate == null) {
            // new Date is initialised to current time
            startDate = new java.util.Date();
        }

        data.put(EVENT_DATE,       event.getEventDate());
        data.put(START_DATE,       startDate);
        // End date can be null
        data.put(END_DATE,         event.getEndDate());
        data.put(START_TIME,       event.getStartTime());
        data.put(END_TIME,         event.getEndTime());
        data.put(LOCATION,         event.getLocation());
        data.put(LEAD,             event.getLead());
        data.put(MAIN_CONTRIBUTOR, event.getMainContributor());
        data.put(EVENT_TYPE,       event.getEventType());
        data.put(MAP_LINK,         event.getMapLink());
        data.put(COST,             event.getCost());
    }

    /** Cancels streamlined editing. */
    public void submitted( FormSectionEvent fse ) {
        if (m_step != null &&
            getSaveCancelSection().getCancelButton()
            .isSelected( fse.getPageState())) {
            m_step.cancelStreamlinedCreation(fse.getPageState());
        }
    }
 
    /** Form processing hook. Saves Event object. */
    public void process(FormSectionEvent fse) {
        FormData data = fse.getFormData();

        Event event
            = (Event) super.processBasicWidgets(fse);

        // save only if save button was pressed
        if (event != null
            && getSaveCancelSection().getSaveButton()
            .isSelected(fse.getPageState())) {

            event.setMainContributor((String) data.get(MAIN_CONTRIBUTOR));
            event.setEventType((String) data.get(EVENT_TYPE));
            event.setMapLink((String) data.get(MAP_LINK));
            event.setEventDate((String) data.get(EVENT_DATE));
            event.setStartDate((java.util.Date) data.get(START_DATE));
            event.setEndDate((java.util.Date) data.get(END_DATE));
            event.setStartTime((java.util.Date) data.get(START_TIME));
            event.setEndTime((java.util.Date) data.get(END_TIME));
            event.setLocation((String) data.get(LOCATION));
            event.setLead((String) data.get(LEAD));
            event.setCost((String) data.get(COST));
            event.save();
        }
        if (m_step != null) {
            m_step.maybeForwardToNextStep(fse.getPageState());
        }
    }
}
