/*
 * Copyright (C) 2002-2004 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the
 * License at http://www.redhat.com/licenses/ccmpl.html.
 *
 * Software distributed under the License is distributed on an
 * "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
 * or implied. See the License for the specific language
 * governing rights and limitations under the License.
 *
 */
package com.arsdigita.cms.contenttypes.ui;


import com.arsdigita.bebop.Component;
import com.arsdigita.bebop.PageState;
import com.arsdigita.cms.ContentItem;
import com.arsdigita.cms.ContentPage;
import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.ItemSelectionModel;
import com.arsdigita.cms.contenttypes.Agenda;
import com.arsdigita.domain.DomainObject;
import com.arsdigita.toolbox.ui.DomainObjectPropertySheet;
import com.arsdigita.cms.ui.authoring.AuthoringKitWizard;
import com.arsdigita.cms.ui.authoring.BasicPageForm;
import com.arsdigita.cms.ui.authoring.SimpleEditStep;
import com.arsdigita.cms.ui.workflow.WorkflowLockedComponentAccess;
import com.arsdigita.cms.util.GlobalizationUtil;

import java.text.DateFormat;

/**
 * Authoring step to edit the simple attributes of the Agenda content type (and
 * its subclasses). The attributes edited are 'name', 'title', ' date' and
 * 'reference code'. This authoring step replaces the
 * <code>com.arsdigita.ui.authoring.PageEdit</code> step for this type.
 **/
public class AgendaPropertiesStep extends SimpleEditStep {

    /** The name of the editing sheet added to this step */
    public static String EDIT_SHEET_NAME = "edit";

    public AgendaPropertiesStep(ItemSelectionModel itemModel,
                                AuthoringKitWizard parent ) {
        super(itemModel, parent );

        setDefaultEditKey(EDIT_SHEET_NAME);
        BasicPageForm editSheet;

        editSheet = new AgendaPropertyForm(itemModel, this);
        add(EDIT_SHEET_NAME, "Edit", new WorkflowLockedComponentAccess(editSheet, itemModel),
            editSheet.getSaveCancelSection().getCancelButton() );

        setDisplayComponent(getAgendaPropertySheet(itemModel));
    }

    /**
     * Returns a component that displays the properties of the Agenda specified
     * by the ItemSelectionModel passed in.
     *
     * @param itemModel The ItemSelectionModel to use
     * @pre itemModel != null
     * @return A component to display the state of the basic properties
     *  of the
     **/
    public static Component getAgendaPropertySheet(ItemSelectionModel
                                                   itemModel ) {
        DomainObjectPropertySheet sheet = new DomainObjectPropertySheet(itemModel);

        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.name"),  Agenda.NAME );
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.title"),  Agenda.TITLE);
        if (!ContentSection.getConfig().getHideLaunchDate()) {
            sheet.add(GlobalizationUtil.globalize("cms.ui.authoring.page_launch_date"),
                      ContentPage.LAUNCH_DATE,
                      new DomainObjectPropertySheet.AttributeFormatter() {
                          public String format(DomainObject item,
                                               String attribute,
                                               PageState state) {
                              ContentPage page = (ContentPage) item;
                              if(page.getLaunchDate() != null) {
                                  return DateFormat.getDateInstance(DateFormat.LONG)
                                      .format(page.getLaunchDate());
                              } else {
                                  return (String)GlobalizationUtil.globalize("cms.ui.unknown").localize();
                              }
                          }
                      });
        }
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.agenda_date_and_time"),  Agenda.AGENDA_DATE);
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.location"),  Agenda.LOCATION);
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.attendees"),  Agenda.ATTENDEES);
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.subject_items"),  Agenda.SUBJECT_ITEMS);
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.contact_info"),  Agenda.CONTACT_INFO);
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.summary"),  Agenda.SUMMARY);
        sheet.add(GlobalizationUtil.globalize("cms.contenttypes.ui.creation_date"),
                  Agenda.AGENDA_DATE,
                  new DomainObjectPropertySheet.AttributeFormatter() {
                      public String format(DomainObject item,
                                           String attribute,
                                           PageState state) {
                          Agenda agenda = (Agenda) item;
                          if(agenda.getCreationDate() != null) {
                              return DateFormat.getDateInstance(DateFormat.LONG)
                                  .format(agenda.getCreationDate());
                          } else {
                              return (String)GlobalizationUtil.globalize("cms.ui.unknown").localize();
                          }
                      }
                  });

        return sheet;
    }
}
