/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 *
 */

package com.arsdigita.populate.cms;

import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.Folder;
import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import com.arsdigita.populate.Utilities;

public class PopulateContent {
    private int m_iParentIndex = 0;
    int m_iContent = 0;
    ContentSection m_section = null;
    String m_sBaseStringSeed = null;
    private final boolean m_manageTransactions;
    
    private static final org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(PopulateContent.class.getName());

    public PopulateContent(final int iContent, 
                           final ContentSection section) throws IllegalArgumentException {

        this(iContent, section, true);
    }

    public PopulateContent(final int iContent, 
                           final ContentSection section,
                           final boolean manageTransactions) throws IllegalArgumentException {
        if (iContent < 0) {
            throw new IllegalArgumentException("Number of Content Items must be >= 0");
        }
        m_iContent = iContent;
        m_section = section;
        m_manageTransactions = manageTransactions;
    }
    
    public void setBaseStringSeed(String sSeed) {
        m_sBaseStringSeed = sSeed;
    }
    
    public String getBaseStringSeed() {
        return m_sBaseStringSeed;
    }

	/**
	     * Populates content.
	     * @param createItem the class for creating the content item
	     * @param parents the list of folders over which to spread the newly created items
	     */
	public void popuate(CreateContentItem createItem, FolderList parents) {
		populate(createItem, parents, 0);
	}

    /**
     * Populates content.
     * @param createItem the class for creating the content item
     * @param parents the list of folders over which to spread the newly created items
     * @param beginParentIndex the index in parents to the first folder to use in population
     */
    public void populate(CreateContentItem createItem, FolderList parents, int beginParentIndex) {
        //populate the content

        Session ses = SessionManager.getSession();
        TransactionContext txn = ses.getTransactionContext();

        String sBase = Utilities.getBaseString(m_sBaseStringSeed);
        int iFolders = parents.size();

        for (int i=0; i < m_iContent; i++) {
            beginTxn();

            String sTitle = createItem.getContentTypeName() + sBase + i;
            s_log.info("Creating content item " + sTitle);

            m_iParentIndex = (beginParentIndex + i) % iFolders;
            Folder parent = parents.get(m_iParentIndex);

            if (s_log.isDebugEnabled()) {
                s_log.debug("Parent of " + sTitle + " is " + parent.getName());
                s_log.debug("Parent folder index is " + m_iParentIndex);
            }

            long start = System.currentTimeMillis();

            createItem.setup(sTitle, parent, m_section);

            long setup = now() - start; start = now();

            createItem.makeContent();

            long makeContent = now() - start; start = now();

            createItem.publish();

            long publish = now() - start; start = now();

            endTxn();

            if (s_log.isDebugEnabled()) {
                s_log.debug("setup() " + setup + " " +
                           "makeContent() " + makeContent + " " +
                           "publish() " + publish + " " +
                           "commitTxn() " + (now() - start));
            }
        }
    }

    private void beginTxn() {
        if (m_manageTransactions) {
            SessionManager.getSession().getTransactionContext().beginTxn();
        }
    }

    private void endTxn() {
        if (m_manageTransactions) {
            SessionManager.getSession().getTransactionContext().commitTxn();
        }
    }

    private static long now() {
        return System.currentTimeMillis();
    }

    /**
     * Returns the index of the last Folder used when populate was last called
     */
    public int getLastFolderIndex() {
        return m_iParentIndex;
    }
}
