/*
 * Copyright (C) 2001, 2002, 2003 Red Hat Inc. All Rights Reserved.
 *
 * The contents of this file are subject to the CCM Public
 * License (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of
 * the License at http://www.redhat.com/licenses/ccmpl.html
 *
 * Software distributed under the License is distributed on an "AS
 * IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * rights and limitations under the License.
 */

package com.arsdigita.populate.cms;

import com.arsdigita.cms.ContentSection;
import com.arsdigita.cms.Folder;
import com.arsdigita.persistence.Session;
import com.arsdigita.persistence.SessionManager;
import com.arsdigita.persistence.TransactionContext;
import com.arsdigita.populate.Utilities;

/**
 * @author bche
 */
public class PopulateFolders {
    private FolderList m_folders;
    private String m_sBaseStringSeed = null;

    private static final org.apache.log4j.Logger s_log =
        org.apache.log4j.Logger.getLogger(PopulateFolders.class.getName());

    public PopulateFolders() {
        m_folders = new FolderList();
    }

    public void setBaseStringSeed(String sSeed) {
        m_sBaseStringSeed = sSeed;
    }

    public String getBaseStringSeed() {
        return m_sBaseStringSeed;
    }

    /**
     * Populates iFolders under the root folder of the content section, section
     * @param iFolders the number of folders to create.  Must be >= 0
     * @param section the content section in which to populate folders 
     * @throws IllegalArgumentException
     */
    public void popuate(int iFolders, ContentSection section)
        throws IllegalArgumentException {
        populate(iFolders, 1, section);
    }

    /**
     * Populates iTopFolders * iFolderDepth folders in the content section, section.  For example, if
     * you specify iTopFolders=5 and iFolderDepth=3, then you will create 5 folders directly under
     * the root folder.  Each of these 5 top-level folders will contain a sub-folder, and a sub-folder of
     * that sub-folder, for a total depth of 3 folders.  
     * @param iRootFolders the number of top-level folders to create.  Must be >=0.
     * @param iFolderDepth the number of levels to nest folders.  Must be >=1.
     * @param section the conten section in which to populate folders
     * @throws IllegalArgumentException
     */
    public void populate(int iTopFolders, int iFolderDepth, ContentSection section)
        throws IllegalArgumentException {
        if (iTopFolders < 0) {
            throw new IllegalArgumentException("Number of Top-level Folders must be >= 0");
        }

        if (iFolderDepth < 1) {
            throw new IllegalArgumentException("Depth of Folders must be >= 1");
        }

        Session ses = SessionManager.getSession();
        TransactionContext txn = ses.getTransactionContext();

        String sFolderDate = Utilities.getBaseString(m_sBaseStringSeed);
        for (int i = 0; i < iTopFolders; i++) {
            txn.beginTxn();

            //create the top-level folder
            Folder folder = new Folder();
            folder.setContentSection(section);
            String sName = "folder" + sFolderDate + i;
            folder.setName(sName);
            folder.setLabel(sName);
            folder.setParent(section.getRootFolder());
            folder.save();
            //place the folder in the ArrayList
            m_folders.add(folder);
            txn.commitTxn();
            s_log.info("Created folder " + sName);

            //create nested folders
            //we subtract 1 from FolderDepth because we just created the top level
            Folder parent = folder;
            Folder child;
            for (int j = 0; j < iFolderDepth - 1; j++) {
                txn.beginTxn();

                //create child
                child = new Folder();
                child.setContentSection(section);
                sName = "folder" + sFolderDate + i + "_" + j;
                child.setName(sName);
                child.setLabel(sName);
                child.setParent(parent);
                child.save();
                //place the folder in the ArrayList
                m_folders.add(child);

                txn.commitTxn();
                s_log.info("Created folder " + sName);

                parent = child;
            }
        }
    }

    /**
     * Retruns an unordered list of all the folders populated.
     * @return an unordered list of all the folders populated
     */
    public FolderList getFolders() {
        return m_folders;
    }
}
